/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.apikit.xml;


import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.Namespace;
import org.jdom2.input.SAXBuilder;
import org.jdom2.output.Format;
import org.jdom2.output.XMLOutputter;
import org.mule.apikit.model.MuleXml;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public abstract class XmlHandler {

  protected final Map<String, Document> inputFiles;


  public XmlHandler(Map<String, InputStream> muleXml) {
    this.inputFiles = fromInputStreams(muleXml);
  }

  public Map<String, Document> getInputFiles() {
    return inputFiles;
  }

  /**
   * @param muleXml map<fileName, muleFileContent>
   * @return the files mapped to scaffolding model
   */
  private Map<String, Document> fromInputStreams(Map<String, InputStream> muleXml) {
    SAXBuilder saxBuilder = getSaxBuilder();

    Map<String, Document> result = new HashMap<>();
    muleXml.entrySet().forEach(entry -> {
      try {
        result.put(entry.getKey(), saxBuilder.build(entry.getValue()));
      } catch (Exception e) {
        throw new RuntimeException("Error reading input files");
      }
    });
    return result;
  }

  /**
   * Receives a list with objects to be add in input XMLs
   * 
   * @param diff represents the result of scaffolding
   * @return
   */
  public abstract Map<String, InputStream> appendDiff(List<MuleXml> diff);

  /**
   * @return a map with namespaces to be add when creating a new mule xml file
   */
  protected abstract Map<String, Namespace> getNamespaces();

  /**
   * Create a new root element <mule></mule> with xml namespaces
   */
  protected Element createMuleXml() {

    Element mule = new Element("mule");

    mule.setNamespace(Namespace.getNamespace("http://www.mulesoft.org/schema/mule/core"));

    mule.addNamespaceDeclaration(Namespace.getNamespace("xsi", "http://www.w3.org/2001/XMLSchema-instance"));

    Map<String, Namespace> namespaces = getNamespaces();

    namespaces.entrySet().forEach(entry -> mule.addNamespaceDeclaration(entry.getValue()));

    mule.setAttribute("schemaLocation", buildSchemaLocation(namespaces),
                      Namespace.getNamespace("xsi", "http://www.w3.org/2001/XMLSchema-instance"));

    return mule;
  }

  /**
   * Transform a root element in XML inputStream
   * 
   * @param element
   * @return
   */
  protected InputStream muleElementToInputStream(Element element) {

    if (element.getDocument() != null) {
      return new ByteArrayInputStream(new XMLOutputter(Format.getPrettyFormat()).outputString(element.getDocument()).getBytes());
    }

    Document output = new Document();
    output.setRootElement(element);
    return new ByteArrayInputStream(new XMLOutputter(Format.getPrettyFormat()).outputString(output).getBytes());

  }

  /**
   * Useful methods for writing schema location in mule xml file
   */
  private String buildSchemaLocation(Map<String, Namespace> namespaces) {
    StringBuilder stringBuilder = new StringBuilder();
    namespaces.entrySet().forEach(entry -> {

      stringBuilder.append(entry.getValue().getURI());
      stringBuilder.append(" ");
      stringBuilder.append(entry.getKey());
      stringBuilder.append(" ");

    });
    return stringBuilder.toString();
  }


  private static SAXBuilder getSaxBuilder() {
    SAXBuilder builder = new SAXBuilder();
    builder.setExpandEntities(false);
    builder.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
    builder.setFeature("http://xml.org/sax/features/external-general-entities", false);
    builder.setFeature("http://xml.org/sax/features/external-parameter-entities", false);
    return builder;
  }
}
