/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.apikit.odata;

import com.google.common.collect.ImmutableList;
import org.apache.olingo.commons.api.edm.Edm;
import org.apache.olingo.commons.api.edm.EdmEntitySet;
import org.apache.olingo.server.api.OData;
import org.apache.olingo.server.api.ServiceMetadata;
import org.mule.apikit.ModelMapper;
import org.mule.apikit.model.Configuration;
import org.mule.apikit.model.ErrorHandler;
import org.mule.apikit.model.Logger;
import org.mule.apikit.model.MuleXml;
import org.mule.apikit.model.OnErrorPropagateFlow;
import org.mule.apikit.model.SetPayloadEE;
import org.mule.apikit.model.SetVariableEE;
import org.mule.apikit.model.Transform;
import org.mule.apikit.odata.model.EntityCollectionListener;
import org.mule.apikit.odata.model.EntityListener;
import org.mule.apikit.odata.model.ListenerConfig;
import org.mule.apikit.odata.model.ListenerOperation;
import org.mule.apikit.odata.model.ODataConfig;
import org.mule.apikit.odata.model.ODataFlow;
import org.mule.apikit.odata.model.RouteOperation;
import org.mule.apikit.odata.model.SerializeEntityCollectionOperation;
import org.mule.apikit.odata.model.SerializeEntityOperation;
import org.mule.apikit.xml.MuleElement;
import org.mule.extension.internal.model.CsdlEdmProvider;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.collect.ImmutableSet.of;
import static java.lang.String.format;
import static java.util.Arrays.asList;

/**
 * Transform OData CSDL to List<MuleXml> representation
 */
public class ODataModelMapper implements ModelMapper {

  private static final Set<String> methods = of("GET", "POST", "PATCH", "PUT", "DELETE");

  @Override
  public List<MuleXml> getModel(File api, Map<String, String> customConfigs) {
    try {
      ServiceMetadata serviceMetadata = OData.newInstance().createServiceMetadata(
                                                                                  new CsdlEdmProvider(new FileInputStream(api)),
                                                                                  new ArrayList<>());

      Edm edm = serviceMetadata.getEdm();

      List<EdmEntitySet> entitySets = edm.getEntityContainer().getEntitySets();

      String fileName = api.getName().substring(0, api.getName().indexOf("."));
      MuleXml muleXml = new MuleXml(format("%s.xml", fileName));

      Configuration configuration = new ODataConfig(format("%s-config", fileName), "api/" + api.getName());
      muleXml.addConfiguration(configuration);

      Configuration listenerConfiguration = new ListenerConfig("HTTP_Listener_Config");
      muleXml.addConfiguration(listenerConfiguration);
      muleXml.addFlow(createMainFlow(configuration, listenerConfiguration));

      entitySets.forEach(entitySet -> {
        List<ODataFlow> flowsForEntitySet = getFlowsForEntitySet(entitySet, configuration);
        flowsForEntitySet.forEach(muleXml::addFlow);
      });

      return ImmutableList.of(muleXml);
    } catch (FileNotFoundException e) {
      throw new RuntimeException("API File Not Found", e);
    }
  }

  /**
   * Generates sources for EntitySet, navigation properties are not considered in scaffolding time(To be reviewed)
   */
  private List<ODataFlow> getFlowsForEntitySet(EdmEntitySet entitySet, Configuration configuration) {
    List<ODataFlow> result = new ArrayList<>();
    methods.forEach(method -> {

      final String flowName = format("%s\\%s\\%s", method, entitySet.getName(), "ENTITY");
      ODataFlow oDataFlow = new ODataFlow(flowName);

      String path = format("/%s", entitySet.getName());

      EntityListener entityListener = new EntityListener(configuration);
      entityListener.setParameter("method", method);
      entityListener.setParameter("path", path);
      oDataFlow.addMuleElement(entityListener);

      Logger logger = new Logger(format("In %s flow", flowName));
      oDataFlow.addMuleElement(logger);
      if (!"DELETE".equals(method)) {
        SerializeEntityOperation serializeEntityOperation = new SerializeEntityOperation(configuration);
        serializeEntityOperation.setParameter("method", method);
        serializeEntityOperation.setParameter("path", path);
        oDataFlow.addMuleElement(serializeEntityOperation);
      }
      result.add(oDataFlow);
    });

    final String flowName = format("%s\\%s\\%s", "GET", entitySet.getName(), "ENTITY_COLLECTION");
    ODataFlow oDataFlow = new ODataFlow(flowName);

    String path = format("/%s", entitySet.getName());

    EntityCollectionListener entityCollectionListener = new EntityCollectionListener(configuration);
    entityCollectionListener.setParameter("method", "GET");
    entityCollectionListener.setParameter("path", path);

    Logger logger = new Logger(format("In %s flow", flowName));

    SerializeEntityCollectionOperation serializeEntityCollectionOperation = new SerializeEntityCollectionOperation(configuration);
    serializeEntityCollectionOperation.setParameter("method", "GET");
    serializeEntityCollectionOperation.setParameter("path", path);

    oDataFlow.addMuleElement(entityCollectionListener);
    oDataFlow.addMuleElement(logger);
    oDataFlow.addMuleElement(serializeEntityCollectionOperation);

    result.add(oDataFlow);
    return result;
  }

  private ODataFlow createMainFlow(Configuration configuration, Configuration listenerConfiguration) {
    // Add HTTP Listener
    ODataFlow mainFlow = new ODataFlow("main-odata-flow");
    ListenerOperation listenerOperation = new ListenerOperation(listenerConfiguration);
    mainFlow.addMuleElement(listenerOperation);
    mainFlow.addMuleElement(getRouteOperation(configuration));
    mainFlow.addMuleElement(getErrorHandler());
    return mainFlow;
  }

  private MuleElement getRouteOperation(Configuration configuration) {
    RouteOperation routeOperation = new RouteOperation(configuration);
    routeOperation.setParameter("method", "#[attributes.method]");
    routeOperation.setParameter("maskedRequestPath", "#[attributes.maskedRequestPath]");
    routeOperation.setParameter("scheme", "#[upper(attributes.scheme)]");
    routeOperation.setParameter("host", "#[attributes.headers.'host']");
    routeOperation.setParameter("listenerPath", "#[attributes.listenerPath]");
    routeOperation.setParameter("httpHeaders", "#[attributes.headers]");
    routeOperation.setParameter("queryString", "#[attributes.queryString]");
    return routeOperation;
  }

  private MuleElement getErrorHandler() {
    OnErrorPropagateFlow onErrorPropagateFlow = new OnErrorPropagateFlow("On_Error_Propagate", true, true, "MULE:ANY");
    SetPayloadEE setPayload = new SetPayloadEE("%dw 2.0\n" +
        "output application/json\n" +
        "---\n" +
        "{\n" +
        "\terror: {\n" +
        "\t\tcode: error.errorMessage.payload.code default \"UNKNOWN\",\n" +
        "\t\tmessage: error.errorMessage.payload.message default error.description,\n" +
        "\t\ttarget: error.errorMessage.payload.target,\n" +
        "\t\tdetails: error.errorMessage.payload.details default [{code: \"UNKNOWN\", message: error.detailedDescription, target: null}],\n"
        +
        "\t\tinnererror: error.errorMessage.payload.innerError default error.childErrors\n" +
        "\t}\n" +
        "}");
    SetVariableEE setVariable =
        new SetVariableEE("statusCode", "error.errorMessage.payload.statusCode default 500");
    Transform transform = new Transform(setPayload, null, asList(setVariable));
    onErrorPropagateFlow.addMuleElement(transform);
    ErrorHandler errorHandler = new ErrorHandler("Error_Handler");
    errorHandler.addOnErrorFlow(onErrorPropagateFlow);
    return errorHandler;
  }
}
