/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.apikit.transform;

import amf.client.model.domain.DomainExtension;
import amf.client.model.domain.FileShape;
import amf.client.model.domain.NodeShape;
import amf.client.model.domain.PropertyShape;
import amf.client.model.domain.ScalarNode;
import amf.client.model.domain.ScalarShape;
import amf.client.model.domain.Shape;
import amf.client.model.domain.UnionShape;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import org.apache.olingo.commons.api.edm.FullQualifiedName;
import org.apache.olingo.commons.api.edm.provider.CsdlEntityContainer;
import org.apache.olingo.commons.api.edm.provider.CsdlEntitySet;
import org.apache.olingo.commons.api.edm.provider.CsdlEntityType;
import org.apache.olingo.commons.api.edm.provider.CsdlProperty;
import org.apache.olingo.commons.api.edm.provider.CsdlPropertyRef;
import org.apache.olingo.commons.api.edm.provider.CsdlSchema;
import org.apache.olingo.commons.core.edm.primitivetype.EdmBinary;
import org.apache.olingo.commons.core.edm.primitivetype.EdmBoolean;
import org.apache.olingo.commons.core.edm.primitivetype.EdmByte;
import org.apache.olingo.commons.core.edm.primitivetype.EdmDate;
import org.apache.olingo.commons.core.edm.primitivetype.EdmDateTimeOffset;
import org.apache.olingo.commons.core.edm.primitivetype.EdmDecimal;
import org.apache.olingo.commons.core.edm.primitivetype.EdmDouble;
import org.apache.olingo.commons.core.edm.primitivetype.EdmGuid;
import org.apache.olingo.commons.core.edm.primitivetype.EdmInt16;
import org.apache.olingo.commons.core.edm.primitivetype.EdmInt32;
import org.apache.olingo.commons.core.edm.primitivetype.EdmInt64;
import org.apache.olingo.commons.core.edm.primitivetype.EdmSingle;
import org.apache.olingo.commons.core.edm.primitivetype.EdmString;
import org.apache.olingo.commons.core.edm.primitivetype.EdmTimeOfDay;
import org.mule.apikit.transform.exception.ODataMetadataFormatException;

import java.util.ArrayList;
import java.util.List;

import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import static org.mule.apikit.transform.RamlConstants.GUID;
import static org.mule.apikit.transform.RamlConstants.INT16;
import static org.mule.apikit.transform.RamlConstants.INT32;
import static org.mule.apikit.transform.RamlConstants.INT64;
import static org.mule.apikit.transform.RamlConstants.INT8;
import static org.mule.apikit.transform.RamlConstants.NAMESPACE_ENTITY_TYPE_NAME;
import static org.mule.apikit.transform.RamlConstants.NAMESPACE_KEY_PROPERTY;
import static org.mule.apikit.transform.RamlConstants.NAMESPACE_NULLABLE_PROPERTY;
import static org.mule.apikit.transform.RamlConstants.NAMESPACE_PRECISION_PROPERTY;
import static org.mule.apikit.transform.RamlConstants.NAMESPACE_REMOTE_NAME;
import static org.mule.apikit.transform.RamlConstants.NAMESPACE_SCALE_PROPERTY;


public class AMFWrapper {

  private static final String AMF_STRING = "http://www.w3.org/2001/XMLSchema#string";
  private static final String AMF_BOOLEAN = "http://www.w3.org/2001/XMLSchema#boolean";
  private static final String AMF_NUMBER = "http://a.ml/vocabularies/shapes#number";
  private static final String AMF_FLOAT = "http://www.w3.org/2001/XMLSchema#float";
  private static final String AMF_DATE_TIME_ONLY = "http://a.ml/vocabularies/shapes#dateTimeOnly";
  private static final String AMF_INTEGER = "http://www.w3.org/2001/XMLSchema#integer";
  private static final String AMF_TIME = "http://www.w3.org/2001/XMLSchema#time";
  private static final String AMF_DATE_TIME = "http://www.w3.org/2001/XMLSchema#dateTime";
  private static final String AMF_DATE_ONLY = "http://www.w3.org/2001/XMLSchema#date";
  private static final String AMF_LONG = "http://www.w3.org/2001/XMLSchema#long";

  private static final Map<String, Function<ScalarShape, FullQualifiedName>> amfToEdm = amfTypeToEdm();
  private static final Map<String, FullQualifiedName> formatToEdm = formatToEdm();

  private static final String ODATA_V4_NAMESPACE = "odata.v4";
  private static final String CONTAINER_NAME = "Container";

  private List<CsdlEntityType> entityTypes = new ArrayList<>();
  private CsdlSchema schema = new CsdlSchema();

  public AMFWrapper(String ramlPath) throws ODataMetadataFormatException {
    List<NodeShape> nodeShapesList = new RAMLSpecParser(ramlPath).getNodeShapes();

    List<CsdlEntitySet> entitySets = new ArrayList<>();
    for (NodeShape nodeShape : nodeShapesList) {
      entitySets.add(buildCSDLEntitySet(nodeShape));
    }

    CsdlEntityContainer entityContainer = new CsdlEntityContainer();
    entityContainer.setName(CONTAINER_NAME);
    entityContainer.setEntitySets(entitySets);
    schema.setEntityContainer(entityContainer);
    schema.setNamespace(ODATA_V4_NAMESPACE);
    schema.setEntityTypes(entityTypes);
  }

  public CsdlSchema getSchema() {
    return schema;
  }


  private CsdlEntitySet buildCSDLEntitySet(NodeShape nodeShape) throws ODataMetadataFormatException {

    if (nodeShape.properties().isEmpty()) {
      throw new ODataMetadataFormatException("No schemas found.");
    }
    CsdlEntitySet entitySet = new CsdlEntitySet();

    String entityName = getAnnotation(nodeShape, NAMESPACE_REMOTE_NAME).orElse(nodeShape.name().value());
    entitySet.setName(entityName);

    CsdlEntityType entityType = buildEntityType(nodeShape);
    entityTypes.add(entityType);
    entitySet.setType(new FullQualifiedName(ODATA_V4_NAMESPACE, entityType.getName()));
    return entitySet;
  }

  private CsdlEntityType buildEntityType(NodeShape nodeShape) throws ODataMetadataFormatException {
    if (nodeShape.properties().isEmpty()) {
      throw new ODataMetadataFormatException("No schemas found.");
    }
    CsdlEntityType entityType = new CsdlEntityType();

    String entityName = getAnnotation(nodeShape, NAMESPACE_ENTITY_TYPE_NAME).orElse(nodeShape.name().value());
    entityType.setName(entityName);

    List<CsdlProperty> csdlProperties = new ArrayList<>();
    List<CsdlPropertyRef> csdlKeys = new ArrayList<>();
    CsdlPropertyRef propertyRef;

    for (PropertyShape propertyShape : nodeShape.properties()) {
      CsdlProperty entityProperty = new CsdlProperty();

      String propertyName = propertyShape.name().value();
      entityProperty.setName(propertyName);

      Shape shape = getScalarShape(propertyShape.range());

      String key = getAnnotation(shape, NAMESPACE_KEY_PROPERTY).orElse(null);
      boolean isKey = Boolean.valueOf(key);
      if (isKey) {
        propertyRef = new CsdlPropertyRef();
        propertyRef.setName(propertyName);
        csdlKeys.add(propertyRef);
      }

      String nullable = getAnnotation(shape, NAMESPACE_NULLABLE_PROPERTY)
          .orElseThrow(() -> new ODataMetadataFormatException(
                                                              format("(odata.nullable) is missing in field : %s, for entity : %s",
                                                                     propertyName, entityName)));

      boolean isNullable = Boolean.valueOf(nullable);

      entityProperty.setNullable(isNullable);

      String defaultValue = (propertyShape.defaultValue() != null ? propertyShape.defaultValue().name().value() : null);
      entityProperty.setDefaultValue(defaultValue);

      if (shape instanceof ScalarShape) {
        ScalarShape scalarShape = (ScalarShape) shape;

        String type = getODataType(scalarShape);

        String maxLength = null;
        if (EdmString.getInstance().getFullQualifiedName().toString().equals(type)) {
          Integer maxLengthInt = scalarShape.maxLength().value();
          maxLength = maxLengthInt != 0 ? String.valueOf(maxLengthInt) : null;

          String unicode = getAnnotation(shape, RamlConstants.NAMESPACE_UNICODE_PROPERTY).orElse(null);
          boolean isUnicode = Boolean.valueOf(unicode);
          entityProperty.setUnicode(isUnicode);
        }

        String precision = getAnnotation(scalarShape, NAMESPACE_PRECISION_PROPERTY).orElse(null);
        String scale = getAnnotation(scalarShape, NAMESPACE_SCALE_PROPERTY).orElse(null);

        if (maxLength != null) {
          entityProperty.setMaxLength(Integer.valueOf(maxLength));
        }
        if (precision != null) {
          entityProperty.setPrecision(Integer.valueOf(precision));
        }
        if (scale != null) {
          entityProperty.setScale(Integer.valueOf(scale));
        }
        entityProperty.setType(type);
      } else if (shape instanceof FileShape) {
        entityProperty.setType(EdmBinary.getInstance().getFullQualifiedName().toString());
      } else {
        throw new ODataMetadataFormatException("Type not supported of property " + propertyName);
      }
      csdlProperties.add(entityProperty);
    }

    entityType.setProperties(csdlProperties);

    if (csdlKeys.isEmpty()) {
      throw new ODataMetadataFormatException("Entity must have a primary key.");
    }
    entityType.setKey(csdlKeys);
    return entityType;
  }

  private String getODataType(ScalarShape scalarShape) {
    String dataType = scalarShape.dataType().value();
    FullQualifiedName name = ofNullable(
                                        amfToEdm.get(dataType).apply(scalarShape))
                                            .orElseThrow(
                                                         () -> new ODataMetadataFormatException(format("Type not supported %s of property %s",
                                                                                                       dataType,
                                                                                                       scalarShape.name())));
    return name.toString();
  }

  private Shape getScalarShape(Shape shape) {
    if (!(shape instanceof UnionShape)) {
      return shape;
    }
    UnionShape unionShape = (UnionShape) shape;
    List<DomainExtension> annotations = shape.customDomainProperties();
    for (Shape unionSubShape : unionShape.anyOf()) {
      if (unionSubShape instanceof ScalarShape) {
        unionSubShape.withCustomDomainProperties(annotations);
        return unionSubShape;
      }
    }
    throw new ODataMetadataFormatException(format("Property %s cannot be just null.", shape.name()));
  }

  private static FullQualifiedName getNumberType(ScalarShape scalarShape) {
    String format = scalarShape.format().value();

    if (format != null) {
      return ofNullable(formatToEdm.get(format))
          .orElseThrow(() -> new ODataMetadataFormatException(format("Unexpected format %s for number type.", format)));
    }

    if (AMF_INTEGER.equals(scalarShape.dataType().value())) {
      return EdmInt32.getInstance().getFullQualifiedName();
    }

    String scale = getAnnotation(scalarShape, NAMESPACE_SCALE_PROPERTY).orElse(null);

    String precision = getAnnotation(scalarShape, NAMESPACE_PRECISION_PROPERTY).orElse(null);

    return scale != null && precision != null ? EdmDecimal.getInstance().getFullQualifiedName()
        : EdmDouble.getInstance().getFullQualifiedName();
  }

  private static FullQualifiedName getStringType(ScalarShape scalarShape) {
    String subType = getAnnotation(scalarShape, RamlConstants.NAMESPACE_TYPE_PROPERTY).orElse(null);
    return GUID.equals(subType) ? EdmGuid.getInstance().getFullQualifiedName() : EdmString.getInstance().getFullQualifiedName();
  }

  /**
   * Extract annotation Value from Shape
   * 
   * @param nodeShape
   * @param annotationName
   * @return empty if annotation not found, annotation value if is present
   */
  private static Optional<String> getAnnotation(Shape nodeShape, String annotationName) {
    Optional<DomainExtension> annotation =
        nodeShape.customDomainProperties().stream()
            .filter(domainExtension -> annotationName.equals(domainExtension.name().value()))
            .findFirst();

    return annotation
        .map(domainExtension -> ((ScalarNode) domainExtension.extension()).value().value());

  }


  private static Map<String, Function<ScalarShape, FullQualifiedName>> amfTypeToEdm() {
    Map<String, Function<ScalarShape, FullQualifiedName>> amfToEdm = new HashMap<>();
    amfToEdm.put(AMF_BOOLEAN, shape -> EdmBoolean.getInstance().getFullQualifiedName());
    amfToEdm.put(AMF_STRING, AMFWrapper::getStringType);
    amfToEdm.put(AMF_FLOAT, shape -> EdmSingle.getInstance().getFullQualifiedName());
    amfToEdm.put(AMF_DATE_TIME_ONLY, shape -> EdmDateTimeOffset.getInstance().getFullQualifiedName());
    amfToEdm.put(AMF_DATE_TIME, shape -> EdmDateTimeOffset.getInstance().getFullQualifiedName());
    amfToEdm.put(AMF_NUMBER, AMFWrapper::getNumberType);
    amfToEdm.put(AMF_INTEGER, AMFWrapper::getNumberType);
    amfToEdm.put(AMF_LONG, AMFWrapper::getNumberType);
    amfToEdm.put(AMF_TIME, shape -> EdmTimeOfDay.getInstance().getFullQualifiedName());
    amfToEdm.put(AMF_DATE_ONLY, shape -> EdmDate.getInstance().getFullQualifiedName());

    return amfToEdm;
  }

  private static Map<String, FullQualifiedName> formatToEdm() {
    Map<String, FullQualifiedName> formatToEdm = new HashMap<>();
    formatToEdm.put(INT64, EdmInt64.getInstance().getFullQualifiedName());
    formatToEdm.put(INT32, EdmInt32.getInstance().getFullQualifiedName());
    formatToEdm.put(INT16, EdmInt16.getInstance().getFullQualifiedName());
    formatToEdm.put(INT8, EdmByte.getInstance().getFullQualifiedName());

    return formatToEdm;
  }
}
