/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.raml.implv2;

import org.apache.commons.io.IOUtils;
import org.mule.raml.implv2.parser.rule.ValidationResultImpl;
import org.mule.raml.implv2.v08.model.RamlImpl08V2;
import org.mule.raml.implv2.v10.model.RamlImpl10V2;
import org.mule.raml.interfaces.model.IRaml;
import org.mule.raml.interfaces.parser.rule.IValidationResult;
import org.raml.v2.api.RamlModelBuilder;
import org.raml.v2.api.RamlModelResult;
import org.raml.v2.api.loader.ResourceLoader;
import org.raml.v2.api.model.common.ValidationResult;
import org.raml.v2.api.model.v10.system.types.AnnotableSimpleType;
import org.raml.v2.internal.impl.RamlBuilder;
import org.raml.v2.internal.impl.commons.nodes.LibraryNodeProvider;
import org.raml.v2.internal.impl.v10.nodes.LibraryLinkNode;
import org.raml.yagi.framework.nodes.Node;
import org.raml.yagi.framework.nodes.snakeyaml.SYIncludeNode;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.mule.raml.implv2.utils.ExchangeDependencyUtils.getExchangeModulePath;

public class ParserV2Utils {

  public static final String PARSER_V2_PROPERTY = "apikit.raml.parser.v2";
  private static final String RAML_PATH_SEPARATOR = "/";

  public static IRaml build(ResourceLoader resourceLoader, String ramlPath) {
    RamlModelResult ramlModelResult = new RamlModelBuilder(resourceLoader).buildApi(ramlPath);
    return wrapApiModel(ramlModelResult, resourceLoader, ramlPath);
  }

  public static IRaml build(ResourceLoader resourceLoader, String ramlPath, String content) {
    RamlModelResult ramlModelResult = new RamlModelBuilder(resourceLoader).buildApi(content, ramlPath);
    return wrapApiModel(ramlModelResult, resourceLoader, ramlPath);
  }

  private static IRaml wrapApiModel(RamlModelResult ramlModelResult, ResourceLoader resourceLoader, String ramlPath) {
    if (ramlModelResult.hasErrors()) {
      throw new RuntimeException("Invalid RAML descriptor.");
    }
    if (ramlModelResult.isVersion08()) {
      return new RamlImpl08V2(ramlModelResult.getApiV08());
    }
    return new RamlImpl10V2(ramlModelResult.getApiV10(), resourceLoader, ramlPath);
  }

  public static List<IValidationResult> validate(ResourceLoader resourceLoader, String ramlPath, String content) {
    List<IValidationResult> result = new ArrayList<>();

    try {
      RamlModelResult ramlApiResult = new RamlModelBuilder(resourceLoader).buildApi(content, ramlPath);
      for (ValidationResult validationResult : ramlApiResult.getValidationResults()) {
        result.add(new ValidationResultImpl(validationResult));
      }
    } catch (Exception e) {
      throw new RuntimeException("Raml parser uncaught exception: " + e.getMessage());
    }
    return result;
  }

  public static List<IValidationResult> validate(ResourceLoader resourceLoader, String ramlPath) {
    return validate(resourceLoader, ramlPath, null);
  }

  public static boolean useParserV2(String content) {
    String property = System.getProperty(PARSER_V2_PROPERTY);
    if (property != null && Boolean.valueOf(property)) {
      return true;
    } else {
      return content.startsWith("#%RAML 1.0");
    }
  }

  public static String nullSafe(AnnotableSimpleType<?> simpleType) {
    return simpleType != null ? String.valueOf(simpleType.value()) : null;
  }

  public static List<String> findIncludeNodes(URI ramlURI, ResourceLoader resourceLoader) throws IOException {
    String rootPath = getParent(ramlURI);
    return findIncludeNodes(rootPath, ramlURI, resourceLoader);
  }

  public static List<String> findIncludeNodes(String rootPath, URI ramlURI, ResourceLoader resourceLoader) throws IOException {
    final InputStream is = resourceLoader.fetchResource(ramlURI.toString());

    if (is == null) {
      return emptyList();
    }

    final Node raml = new RamlBuilder().build(IOUtils.toString(is));
    return findIncludeNodes(rootPath, raml, resourceLoader);
  }

  public static List<String> findIncludeNodes(String rootPath, final Node raml, ResourceLoader resourceLoader)
      throws IOException {
    final Set<String> includePaths = new HashSet<>();
    findIncludeNodes(rootPath, "", includePaths, singletonList(raml), resourceLoader);
    return new ArrayList<>(includePaths);
  }

  private static void findIncludeNodes(String rootPath, final String pathRelativeToRoot, final Set<String> includePaths,
                                       final List<Node> currents, ResourceLoader resourceLoader)
      throws IOException {

    for (final Node current : currents) {
      // search for include in sources of the current node
      Node possibleInclude = current;
      String pathRelativeToRootCurrent = pathRelativeToRoot;
      while (possibleInclude != null) {
        String includePath = null;
        if (possibleInclude instanceof SYIncludeNode) {
          includePath = ((SYIncludeNode) possibleInclude).getIncludePath();
        } else if (possibleInclude instanceof LibraryLinkNode) {
          includePath = ((LibraryLinkNode) possibleInclude).getRefName();
        }

        if (includePath != null) {
          final String absolutIncludePath = computeIncludePath(rootPath, pathRelativeToRoot, includePath);
          final URI includedFileAsUri = URI.create(absolutIncludePath).normalize();
          includePaths.add(includedFileAsUri.toString());
          includePaths.addAll(findIncludeNodes(rootPath, includedFileAsUri, resourceLoader));
          pathRelativeToRootCurrent = calculateNextRootRelative(pathRelativeToRootCurrent,
                                                                includePath);
        }

        possibleInclude = possibleInclude.getSource();
      }

      findIncludeNodes(rootPath, pathRelativeToRootCurrent, includePaths, getChildren(current), resourceLoader);
    }
  }

  private static String getParent(URI uri) {
    final URI parentUri = uri.getPath().endsWith("/") ? uri.resolve("..") : uri.resolve(".");
    final String parentUriAsString = parentUri.toString();
    return parentUriAsString.endsWith("/") ? parentUriAsString.substring(0, parentUriAsString.length() - 1) : parentUriAsString;
  }

  private static String fixExchangeModulePath(String path) {
    return getExchangeModulePath(path);
  }

  private static String calculateNextRootRelative(String pathRelativeToRootCurrent, String includePath) {
    String newRelativeSubPath = getParent(URI.create(includePath));
    newRelativeSubPath = newRelativeSubPath == null ? "" : newRelativeSubPath;
    return pathRelativeToRootCurrent + newRelativeSubPath;
  }

  private static List<Node> getChildren(Node node) {
    if (node instanceof LibraryLinkNode) {
      node = ((LibraryLinkNode) node).getRefNode();
    }
    List<Node> result = new ArrayList<>();
    if (node != null) {
      if (node instanceof LibraryNodeProvider) {
        LibraryNodeProvider libraryNodeProvider = (LibraryNodeProvider) node;
        Node libraryNode = libraryNodeProvider.getLibraryNode();
        if (libraryNode != null) {
          result.add(libraryNode);
        }
      }
      result.addAll(node.getChildren());
    }

    return result;
  }

  private static String computeIncludePath(final String rootPath, final String pathRelativeToRoot, final String includePath) {
    // according to RAML 1.0 spec: https://github.com/raml-org/raml-spec/blob/master/versions/raml-10/raml-10.md

    // uses File class to normalize the resolved path acording with the OS (every slash in the path must be in the same direction)
    final String absolutePath = isAbsolute(includePath) //
        ? rootPath + includePath
        // relative path: A path that neither begins with a single slash ("/") nor constitutes a URL, and is interpreted relative to the location of the included file.
        : rootPath + (pathRelativeToRoot.isEmpty() ? "" : "/" + pathRelativeToRoot) + "/" + includePath;
    return fixExchangeModulePath(absolutePath);
  }

  private static boolean isAbsolute(String includePath) {
    // absolute path: A path that begins with a single slash ("/") and is interpreted relative to the root RAML file location.
    return includePath.startsWith(RAML_PATH_SEPARATOR);
  }

}
