package org.mule.certification;

/*
 * Copyright 2001-2005 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Execute;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.mule.certification.verifier.Verifier;

import java.io.File;
import java.io.IOException;
import java.util.*;

import static org.mule.certification.verifier.VerifiersFactory.*;

/**
 * Validates a connector project structure is the expected for
 * a certifiable connector.
 */
@Mojo( name = "validate")
@Execute( goal = "validate")
public class StructureMojo extends AbstractMojo {

    public static final String CLOSED_HEADER_MODEL = "closedHeader.txt";
    public static final String OPEN_HEADER_MODEL = "openHeader.txt";
    public static final String CLOSED_LICENSE_MODEL = "closedLicense.txt";
    public static final String OPEN_LICENSE_MODEL = "openLicense.txt";
    public static final String REPORT_DIR = "reports";
    public static final String REPORT_FILE = "certificationReport.json";

    @Parameter( defaultValue = "${project.basedir}", readonly = true )
    protected File basedir;

    @Parameter( defaultValue = "${project.build.directory}", readonly = true )
    protected File target;

    @Parameter(property = "closedProject", defaultValue = "false", readonly = true )
    private Boolean closedProject;

    @Parameter(property = "checkLicenseContent", defaultValue = "false", readonly = true )
    private Boolean checkLicenseContent;

    private Report report = new Report();

    private enum RequiredKey { DOC, DEMO, ICONS, MAIN, TEST, HEADER, LICENSE_MD, README_MD }

    public void execute() throws MojoExecutionException, MojoFailureException {
        setLogLevel();

        Map<RequiredKey, File> requiredResources = new HashMap<RequiredKey, File>(){{
            put(RequiredKey.DOC, new File(basedir, "doc"));
            put(RequiredKey.DEMO, new File(basedir, "demo"));
            put(RequiredKey.ICONS, new File(basedir, "icons"));
            put(RequiredKey.MAIN, new File(basedir, "src/main/java"));
            put(RequiredKey.TEST, new File(basedir, "src/test/java"));
            put(RequiredKey.README_MD, new File(basedir, "README.md"));
            put(RequiredKey.LICENSE_MD, new File(basedir, "LICENSE.md"));
            put(RequiredKey.HEADER, new File(basedir, "LICENSE_HEADER.txt"));
        }};

        List<Verifier> verifiers = setupVerifiers(requiredResources);

        boolean failed = runVerifiers(verifiers);

        exportReport();

        if (failed){
            throw new MojoFailureException(
                        String.format("Failed to verify some rules, please check the full results in target/%s/%s", REPORT_DIR, REPORT_FILE));
        }

    }

    private boolean runVerifiers(List<Verifier> verifiers) {
        boolean failed = false;
        for(Verifier v: verifiers){
            try {
                if (!v.verify()){
                    logFailure(v);
                    failed = true;
                } else {
                    logSuccess(v);
                }
            } catch (IOException e) {
                getLog().debug(e.getMessage());
                logFailure(v);
            }
        }
        return failed;
    }

    private void exportReport() {
        try {
            File destination = new File(target, REPORT_DIR);
            FileUtils.forceMkdir(destination);
            FileUtils.write(new File(destination, REPORT_FILE), report.export());
        } catch (IOException e) {
            e.printStackTrace();
            getLog().error("Unable to export results to target/" + REPORT_FILE);
        }
    }

    private List<Verifier> setupVerifiers(Map<RequiredKey, File> requiredResources) {
        List<Verifier> verifiers = new LinkedList<Verifier>();

        for(File required: requiredResources.values()){
            verifiers.add(existsAndNotEmpty(required));
        }

        verifiers.add(existsInPath(requiredResources.get(RequiredKey.DOC), ".+\\.md", "ending with '.md'"));
        verifiers.add(existsInPath(requiredResources.get(RequiredKey.ICONS), ".*24x16\\.png", "ending with '24x16.png'"));
        verifiers.add(existsInPath(requiredResources.get(RequiredKey.ICONS), ".*48x32\\.png", "ending with '48x32.png'"));
        verifiers.add(existsInPath(requiredResources.get(RequiredKey.MAIN), ".+Connector.java", "ending with 'Connector.java'"));

        if (checkLicenseContent) {
            verifiers.add(hasContent(requiredResources.get(RequiredKey.LICENSE_MD), getLicenseContent()));
            verifiers.add(hasContent(requiredResources.get(RequiredKey.HEADER), getHeaderContent()));
        }
        return verifiers;
    }

    private void logSuccess(Verifier v) {
        getLog().debug("Verified rule :: " + v.ruleMessage());
        report.passed(v.ruleMessage());
    }

    private void logFailure(Verifier v) {
        getLog().error("Failed to verify rule :: " + v.ruleMessage());
        report.failed(v.ruleMessage());
    }

    private List<String> getHeaderContent() {
        try {
            String name = closedProject ? CLOSED_HEADER_MODEL : OPEN_HEADER_MODEL;
            return IOUtils.readLines(Thread.currentThread().getContextClassLoader().getResourceAsStream(name));
        } catch (IOException e) {
            e.printStackTrace();
        }
        return new ArrayList<String>();
    }

    private List<String> getLicenseContent() {
        try {
            String name = closedProject ? CLOSED_LICENSE_MODEL : OPEN_LICENSE_MODEL;
            return IOUtils.readLines(Thread.currentThread().getContextClassLoader().getResourceAsStream(name));
        } catch (IOException e) {
            e.printStackTrace();
        }
        return new ArrayList<String>();
    }

    private void setLogLevel() {
        if (getLog().isDebugEnabled())
            Logger.getRootLogger().setLevel(Level.DEBUG);
    }
}
