package org.mule.commons.atlantic.execution.builder;

import org.mule.commons.atlantic.execution.context.ExecutionContext;
import org.mule.commons.atlantic.lambda.AtlanticLambda;

import java.util.List;

/**
 * Parent of all the execution builders. This class provides the basic structure for all the ExecutionBuilders to work.
 *
 * An ExecutionBuilder takes a determined type of {@link AtlanticLambda} and works it's way down by adding  parameters
 * to it and converting into the next type of ExecutionBuilder that handles the {@link AtlanticLambda} with one less
 * parameter. This ends when there are no more parameters and the last ExecutionBuilder, an instance of
 * {@link FunctionExecutionBuilder} or {@link SupplierExecutionBuilder}, gets the
 * {@link org.mule.commons.atlantic.execution.context.executor.Executor} from the ExecutionContext and executes the
 * lambda on it.
 *
 * An example of how ExecutionBuilders work is the following:
 *
 * <code>
 *     TriFunctionExecutionBuilder{@literal <}MyParam1Type, MyParam2Type, MyParam3Type, MyResultType, MyResultType{@literal >} original = new TriFunctionExecutionBuilder{@literal <}{@literal >}(MyClass::staticMethodWith3Params, new ArrayList{@literal <}{@literal >}(), Parser.self(), myExecutionContext);
 *     FunctionExecutionBuilder{@literal <}MyParam2Type, MyParam3Type, MyResultType, MyResultType{@literal >} next = new TriFunctionExecutionBuilder{@literal <}{@literal >}(MyClass::staticMethodWith3Params, new ArrayList{@literal <}{@literal >}(), Parser.self(), myExecutionContext).withParam(param1);
 *     FunctionExecutionBuilder{@literal <}MyParam3Type, MyResultType, MyResultType{@literal >} last = next.withParam(param2);
 *     MyResultType result = last.withParam(param3);
 * </code>
 *
 * To simplify that block of code, the use is the following:
 *
 * <code>
 *     MyResultType result = new TriFunctionExecutionBuilder{@literal <}{@literal >}(MyClass::staticMethodWith3Params, new ArrayList{@literal <}{@literal >}(), Parser.self(), myExecutionContext).withParam(param1)
 *         .withParam(param2)
 *         .withParam(param3);
 * </code>
 *
 * This class also contains an {@link ExecutionContext}, which will provide functionality over the execution of the
 * lambda.
 *
 * @param <L> The type of {@link AtlanticLambda} to handle.
 * @param <RESULT> The type of result returned by the execution.
 * @see ExecutionContext
 */
public class ExecutionBuilder<L extends AtlanticLambda, RESULT> {

    private final L lambda;
    private final List<Object> params;
    private final ExecutionContext<RESULT> context;

    /**
     * Default constructor.
     *
     * @param lambda       The {@link AtlanticLambda} to execute.
     * @param params       The parameters already accumulated when creating this ExecutionBuilder.
     * @param context      The {@link ExecutionContext} that contains all the listeners and handlers of the execution.
     */
    protected ExecutionBuilder(L lambda, List<Object> params, ExecutionContext<RESULT> context) {
        this.lambda = lambda;
        this.params = params;
        this.context = context;
    }

    /**
     * Gets the {@link AtlanticLambda} to execute.
     *
     * @return AtlanticLambda The lambda to execute.
     */
    protected L getLambda() {
        return lambda;
    }

    /**
     * Gets the list of parameters accumulated for the execution.
     *
     * @return List The list.
     */
    protected List<Object> getParams() {
        return params;
    }

    /**
     * Gets the {@link ExecutionContext} that contains all the listeners and handlers of the execution.
     *
     * @return ExecutionContext The context.
     */
    protected ExecutionContext<RESULT> getContext() {
        return context;
    }
}
