package org.mule.commons.atlantic.execution.builder;

import org.mule.commons.atlantic.exception.ConversionException;
import org.mule.commons.atlantic.execution.context.ExecutionContext;
import org.mule.commons.atlantic.execution.parser.Parser;
import org.mule.commons.atlantic.lambda.function.Function;

import java.util.List;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toList;

/**
 * {@link ExecutionBuilder} that does the same as a {@link GenericFunctionExecutionBuilder} but instead of returning a
 * {@link SupplierExecutionBuilder} on the with param, builds it and calls {@link SupplierExecutionBuilder#execute()} on
 * it.
 *
 * @param <A>      The only parameter of the {@link Function}.
 * @param <RESULT> The type of the result of the {@link Function}.
 */
public class FunctionExecutionBuilder<A, RESULT> extends ExecutionBuilder<Function<A, RESULT>, RESULT> {
    /**
     * Default constructor.
     *
     * @param lambda  The {@link Function} to execute.
     * @param params  The parameters already accumulated when creating this ExecutionBuilder.
     * @param context The {@link ExecutionContext} that contains all the listeners and handlers of the execution.
     */
    public FunctionExecutionBuilder(Function<A, RESULT> lambda, List<Object> params, ExecutionContext context) {
        super(lambda, params, context);
    }

    /**
     * Downgrades the {@link FunctionExecutionBuilder} to a {@link SupplierExecutionBuilder} by completing the last
     * parameter and then calls {@link SupplierExecutionBuilder#execute()} on it. Then returns the result of the
     * execution.
     *
     * @param param The last parameter.
     * @return Object The result of the execution.
     */
    public RESULT withParam(A param) {
        return new SupplierExecutionBuilder<>(() -> getLambda().apply(param),
                Stream.concat(getParams().stream(), Stream.of(param)).collect(toList()),
                getContext()).execute();
    }

    /**
     * As {@link #withParam(Object)} but converts the input to the required value of the param using a {@link Parser}.
     *
     * @param input   The unparsed last parameter.
     * @param parser  The {@link Parser} that converts from the unparsed input to the required parameter of the execution.
     * @param <INPUT> The type of the unparsed input.
     * @return Object The result of the execution.
     */
    public <INPUT> RESULT withParam(INPUT input, Parser<INPUT, A> parser) {
        A param;
        try {
            param = parser.apply(input);
        } catch (Throwable throwable) {
            throw new ConversionException(input, throwable);
        }
        return withParam(param);
    }
}
