package org.mule.commons.atlantic.execution.builder.factory;

import org.mule.commons.atlantic.exception.UnhandledException;
import org.mule.commons.atlantic.execution.context.ExecutionContext;
import org.mule.commons.atlantic.execution.context.exception.DefinedExceptionHandler;
import org.mule.commons.atlantic.execution.context.exception.ExceptionHandler;
import org.mule.commons.atlantic.execution.context.exception.PassThroughExceptionHandler;
import org.mule.commons.atlantic.execution.context.executor.BlockingExecutor;
import org.mule.commons.atlantic.execution.context.executor.Executor;
import org.mule.commons.atlantic.execution.context.listener.PostExecutionListener;
import org.mule.commons.atlantic.execution.context.listener.PreExecutionListener;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

/**
 * Parent class of the {@link InstanceExecutionBuilderFactory} and {@link StaticExecutionBuilderFactory} classes. Allows the child classes to build
 * {@link ExecutionContext} instances to include on the execution build.
 *
 * @param <SELF> The type of sublclass of the ExecutionBuilderFactory. This will be returned by each builder method declared here.
 * @param <RESULT> The type of result returned by the execution to build.
 * @see Executor
 * @see PreExecutionListener
 * @see PostExecutionListener
 * @see DefinedExceptionHandler
 * @see ExceptionHandler
 */
public class ExecutionBuilderFactory<SELF extends ExecutionBuilderFactory, RESULT> {
    private Executor executor = new BlockingExecutor();
    private List<PreExecutionListener> preExecutionListeners = new ArrayList<>();
    private List<PostExecutionListener> postExecutionListeners = new ArrayList<>();
    private List<DefinedExceptionHandler<?>> exceptionHandlers = new ArrayList<>();

    /**
     * Defines the {@link Executor} that will handle the threads used on the running of the method.
     *
     * @param executor The handler to set.
     * @return ExecutionBuilderFactory This execution builder factory.
     */
    public SELF withExecutor(Executor<RESULT> executor) {
        this.executor = executor;
        return (SELF) this;
    }

    /**
     * Adds an array of {@link PreExecutionListener}s to be executed before the execution.
     *
     * @param preExecutionListeners The listeners to set.
     * @return ExecutionBuilderFactory This execution builder factory.
     */
    public SELF withPreExecutionListener(PreExecutionListener... preExecutionListeners) {
        return addToList(this.preExecutionListeners, Stream.of(preExecutionListeners));
    }

    /**
     * Adds a list of {@link PreExecutionListener}s to be executed before the execution.
     *
     * @param preExecutionListeners The listeners to set.
     * @return ExecutionBuilderFactory This execution builder factory.
     */
    public SELF withPreExecutionListener(List<PreExecutionListener> preExecutionListeners) {
        return addToList(this.preExecutionListeners, preExecutionListeners.stream());
    }

    /**
     * Adds an array of {@link PostExecutionListener}s to be executed after the execution.
     *
     * @param postExecutionListeners The listeners to set.
     * @return ExecutionBuilderFactory This execution builder factory.
     */
    public SELF withPostExecutionListener(PostExecutionListener... postExecutionListeners) {
        return addToList(this.postExecutionListeners, Stream.of(postExecutionListeners));
    }

    /**
     * Adds a list of {@link PostExecutionListener}s to be executed after the execution.
     *
     * @param postExecutionListeners The listeners to set.
     * @return ExecutionBuilderFactory This execution builder factory.
     */
    public SELF withPostExecutionListener(List<PostExecutionListener> postExecutionListeners) {
        return addToList(this.postExecutionListeners, postExecutionListeners.stream());
    }

    /**
     * Adds an {@link ExceptionHandler} to handle a specific kind of {@link Throwable} or it's subclasses. Handlers are
     * triggered in the order they were set into the context with only the first match found triggering. This means that
     * if the following case would happen:
     *
     * <code>
     *     executor.withExceptionHandler(Throwable.class, myThrowableExceptionHandler)
     *         .withExceptionHandler(RuntimeException.class, myRuntimeExceptionExceptionHandler);
     * </code>
     *
     * And a {@link RuntimeException} is thrown, then the myThrowableExceptionHandler would catch it and handle it.
     *
     * Additionally, any non handled exceptions, unless ignored using {@link ExecutionBuilderFactory#withIgnoredExceptionType(Class)},
     * will be wrapped in an {@link UnhandledException} and thrown as such.
     *
     * @param exceptionClass   The exception type to handle. This will include the specific exception and all it's
     *                         subclasses.
     * @param exceptionHandler The handler for the exception.
     * @param <T>              The subtype of {@link Throwable} that will be handled here.
     * @return ExecutionBuilderFactory This execution builder factory.
     */
    public <T extends Throwable> SELF withExceptionHandler(Class<T> exceptionClass, ExceptionHandler<T> exceptionHandler) {
        return withExceptionHandler(new DefinedExceptionHandler<>(exceptionClass, exceptionHandler));
    }

    /**
     * Adds a {@link DefinedExceptionHandler} to handle a specific kind of {@link Throwable} or it's subclasses. Handlers are
     * triggered in the order they were set into the context with only the first match found triggering. This means that
     * if the following case would happen:
     *
     * <code>
     *     executor.{@literal <}Throwable{@literal >}withExceptionHandler(myDefinedThrowableExceptionHandler)
     *         .{@literal <}RuntimeException{@literal >}withExceptionHandler(myDefinedRuntimeExceptionExceptionHandler);
     * </code>
     *
     * And a {@link RuntimeException} is thrown, then the myDefinedThrowableExceptionHandler would catch it and handle it.
     *
     * Additionally, any non handled exceptions, unless ignored using {@link ExecutionBuilderFactory#withIgnoredExceptionType(Class)},
     * will be wrapped in an {@link UnhandledException} and thrown as such.
     *
     * @param exceptionHandler The handler for the exception.
     * @param <T>              The subtype of {@link Throwable} that will be handled here.
     * @return ExecutionBuilderFactory This execution builder factory.
     */
    public <T extends Throwable> SELF withExceptionHandler(DefinedExceptionHandler<T> exceptionHandler) {
        return addToList(exceptionHandlers, Stream.of(exceptionHandler));
    }

    /**
     * Allows a specific {@link RuntimeException} and it's subclasses to be ignored by the exception handlers.
     *
     * @param exceptionClass The class of the runtime exception to ignore. it must extend {@link RuntimeException}
     * @param <T>            The type of the {@link RuntimeException} to ignore.
     * @return ExecutionBuilderFactory This execution builder factory.
     */
    public <T extends RuntimeException> SELF withIgnoredExceptionType(Class<T> exceptionClass) {
        Optional.ofNullable(exceptionClass).map(PassThroughExceptionHandler::new).ifPresent(handler -> exceptionHandlers.add(0, handler));
        return (SELF) this;
    }

    private <T> SELF addToList(List<T> list, Stream<T> elementsToAdd) {
        elementsToAdd.forEach(list::add);
        return (SELF) this;
    }

    /**
     * Builds the {@link ExecutionContext}.
     *
     * @return ExecutionContext The execution context built based on this builder.
     */
    protected ExecutionContext<RESULT> build() {
        return new ExecutionContext(executor, preExecutionListeners, postExecutionListeners, exceptionHandlers);
    }
}
