package org.mule.commons.atlantic.execution.context;

import org.mule.commons.atlantic.execution.context.exception.DefinedExceptionHandler;
import org.mule.commons.atlantic.execution.context.executor.Executor;
import org.mule.commons.atlantic.execution.context.listener.PostExecutionListener;
import org.mule.commons.atlantic.execution.context.listener.PreExecutionListener;
import org.mule.commons.atlantic.lambda.supplier.Supplier;

import java.util.List;

import static java.util.Collections.unmodifiableList;

/**
 * Class that contains the listeners and handlers for a determined execution.
 *
 * @param <RESULT> The type of result returned by the execution.
 * @see Executor
 * @see PreExecutionListener
 * @see PostExecutionListener
 * @see DefinedExceptionHandler
 */
public class ExecutionContext<RESULT> {
    private final Executor<RESULT> executor;
    private final List<PreExecutionListener> preExecutionListeners;
    private final List<PostExecutionListener<RESULT>> postExecutionListeners;
    private final List<DefinedExceptionHandler<?>> exceptionHandlers;

    /**
     * Default constructor. Sets the contents of this class.
     *
     * @param executor               The handler class that will deal with the execution of the operation.
     * @param preExecutionListeners  The listeners that will be triggered before executing the operation.
     * @param postExecutionListeners The listeners that will be triggered after the execution of the operation.
     * @param exceptionHandlers      The classes that will be handling the exeptions thrown during the execution.
     */
    public ExecutionContext(Executor<RESULT> executor,
                            List<PreExecutionListener> preExecutionListeners,
                            List<PostExecutionListener<RESULT>> postExecutionListeners,
                            List<DefinedExceptionHandler<?>> exceptionHandlers) {
        this.executor = executor;
        this.preExecutionListeners = unmodifiableList(preExecutionListeners);
        this.postExecutionListeners = unmodifiableList(postExecutionListeners);
        this.exceptionHandlers = unmodifiableList(exceptionHandlers);
    }

    /**
     * Executes a {@link Supplier} over it's {@link Executor}.
     *
     * @param operation The supplier to execute.
     * @param params    The accumulated params for the supplier.
     * @return Object The result of the lambda execution.
     */
    public RESULT execute(Supplier<RESULT> operation, List<Object> params) {
        return executor.execute(operation, params, preExecutionListeners, postExecutionListeners, exceptionHandlers);
    }
}
