package org.mule.commons.atlantic.lambda.function;

import org.mule.commons.atlantic.lambda.supplier.Supplier;

import java.util.Objects;

/**
 * Represents a function that accepts one argument and produces a result.
 *
 * This is a functional interface whose functional method is {@link #apply(Object)}.
 *
 * @param <A>      the type of the input to the function
 * @param <RESULT> the type of the result of the function
 */
@FunctionalInterface
public interface Function<A, RESULT> extends AtlanticFunction<A, Supplier<RESULT>> {

    /**
     * Applies this function to the given argument.
     *
     * @param parameter The function argument.
     * @return Object The function result.
     * @throws Throwable Any exception that the operation will throw.
     */
    RESULT apply(A parameter) throws Throwable;

    @Override
    default Supplier<RESULT> downgrade(A a) {
        return () -> apply(a);
    }

    /**
     * Returns a composed function that first applies this function to its input, and then applies the {@code after}
     * function to the result.
     * If evaluation of either function throws an exception, it is relayed to the caller of the composed function.
     *
     * @param <AFTER_RESULT> The type of output of the {@code after} function, and of the composed function.
     * @param after          The function to apply after this function is applied.
     * @return Function A composed function that first applies this function and then applies the {@code after} function
     */
    default <AFTER_RESULT> Function<A, AFTER_RESULT> andThen(Function<? super RESULT, ? extends AFTER_RESULT> after) {
        Objects.requireNonNull(after);
        return (A t) -> after.apply(apply(t));
    }
}
