package org.mule.commons.atlantic.lambda.function;

import java.util.Objects;


/**
 * Represents a function that accepts four arguments and produces a result.
 * This is the four-arity specialization of {@link Function}.
 *
 * This is a functional interface whose functional method is {@link #apply(Object, Object, Object, Object)}.
 *
 * @param <A>      The type of the first argument to the function
 * @param <B>      The type of the second argument to the function
 * @param <C>      The type of the third argument to the function
 * @param <D>      The type of the fourth argument to the function
 * @param <RESULT> The type of the result of the function
 */
@FunctionalInterface
public interface TetraFunction<A, B, C, D, RESULT> extends AtlanticFunction<A, TriFunction<B, C, D, RESULT>> {

    /**
     * Applies this function to the given arguments.
     *
     * @param a The first function argument.
     * @param b The second function argument.
     * @param c The third function argument.
     * @param d The fourth function argument.
     * @return Object The function result.
     * @throws Throwable Any exception that the operation will throw.
     */
    RESULT apply(A a, B b, C c, D d) throws Throwable;

    @Override
    default TriFunction<B, C, D, RESULT> downgrade(A a) {
        return (b, c, d) -> apply(a, b, c, d);
    }

    /**
     * Returns a composed tetrafunction that first applies this tetrafunction to its input, and then applies the {@code after}
     * function to the result.
     * If evaluation of either function throws an exception, it is relayed to the caller of the composed function.
     *
     * @param <AFTER_RESULT> The type of output of the {@code after} function, and of the composed function.
     * @param after          The function to apply after this function is applied.
     * @return TetraFunction A composed tetrafunction that first applies this function and then applies the {@code after} function
     */
    default <AFTER_RESULT> TetraFunction<A, B, C, D, AFTER_RESULT> andThen(Function<? super RESULT, ? extends AFTER_RESULT> after) {
        Objects.requireNonNull(after);
        return (A a, B b, C c, D d) -> after.apply(apply(a, b, c, d));
    }
}
