package org.mule.commons.atlantic.execution.builder.factory;

import org.mule.commons.atlantic.exception.UnhandledException;
import org.mule.commons.atlantic.execution.ExecutionFactory;
import org.mule.commons.atlantic.execution.exception.handler.DefinedExceptionHandler;
import org.mule.commons.atlantic.execution.exception.handler.ExceptionHandler;
import org.mule.commons.atlantic.execution.exception.handler.PassThroughExceptionHandler;
import org.mule.commons.atlantic.execution.listener.OnPreExecutionListener;
import org.mule.commons.atlantic.execution.listener.OnSuccessListener;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

/**
 * Parent class of the {@link InstanceExecutionFactoryBuilder} and {@link StaticExecutionFactoryBuilder} classes. Allows the child classes to buildExecutionFactory
 * {@link ExecutionFactory} instances to include on the execution buildExecutionFactory.
 *
 * @param <SELF> The type of sublclass of the DefaultExecutionFactoryBuilder. This will be returned by each builder method declared here.
 * @see DefinedExceptionHandler
 * @see ExceptionHandler
 */
public class DefaultExecutionFactoryBuilder<SELF extends DefaultExecutionFactoryBuilder> implements ExecutionFactoryBuilder<SELF> {
    private List<OnPreExecutionListener> onPreExecutionListeners = new ArrayList<>();
    private List<OnSuccessListener> onSuccessListeners= new ArrayList<>();
    private List<DefinedExceptionHandler<?>> exceptionHandlers = new ArrayList<>();

    /**
     * Adds an array of {@link OnPreExecutionListener}s to be executed before the execution.
     *
     * @param onPreExecutionListeners The listeners to set.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    public SELF withPreExecutionListener(OnPreExecutionListener... onPreExecutionListeners) {
        return addToList(this.onPreExecutionListeners, Stream.of(onPreExecutionListeners));
    }

    /**
     * Adds a list of {@link OnPreExecutionListener}s to be executed before the execution.
     *
     * @param onPreExecutionListeners The listeners to set.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    public SELF withPreExecutionListener(List<OnPreExecutionListener> onPreExecutionListeners) {
        return addToList(this.onPreExecutionListeners, onPreExecutionListeners.stream());
    }

    /**
     * Adds an array of {@link OnPreExecutionListener}s to be executed after the execution.
     *
     * @param onSuccessListeners The listeners to set.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    public SELF withPostExecutionListener(OnSuccessListener... onSuccessListeners) {
        return addToList(this.onSuccessListeners, Stream.of(onSuccessListeners));
    }

    /**
     * Adds a list of {@link OnPreExecutionListener}s to be executed after the execution.
     *
     * @param postExecutionListeners The listeners to set.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    public SELF withPostExecutionListener(List<OnSuccessListener> postExecutionListeners) {
        return addToList(this.onSuccessListeners, onSuccessListeners.stream());
    }

    /**
     * Adds an {@link ExceptionHandler} to handle a specific kind of {@link Throwable} or it's subclasses. Handlers are
     * triggered in the order they were set into the context with only the first match found triggering. This means that
     * if the following case would happen:
     *
     * <code>
     *     executor.withExceptionHandler(Throwable.class, myThrowableExceptionHandler)
     *         .withExceptionHandler(RuntimeException.class, myRuntimeExceptionExceptionHandler);
     * </code>
     *
     * And a {@link RuntimeException} is thrown, then the myThrowableExceptionHandler would catch it and handle it.
     *
     * Additionally, any non handled exceptions, unless ignored using {@link DefaultExecutionFactoryBuilder#withIgnoredExceptionType(Class)},
     * will be wrapped in an {@link UnhandledException} and thrown as such.
     *
     * @param exceptionClass   The exception type to handle. This will include the specific exception and all it's
     *                         subclasses.
     * @param exceptionHandler The handler for the exception.
     * @param <T>              The subtype of {@link Throwable} that will be handled here.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    public <T extends Throwable> SELF withExceptionHandler(Class<T> exceptionClass, ExceptionHandler<T> exceptionHandler) {
        return withExceptionHandler(new DefinedExceptionHandler<>(exceptionClass, exceptionHandler));
    }

    /**
     * Adds a {@link DefinedExceptionHandler} to handle a specific kind of {@link Throwable} or it's subclasses. Handlers are
     * triggered in the order they were set into the context with only the first match found triggering. This means that
     * if the following case would happen:
     *
     * <code>
     *     executor.{@literal <}Throwable{@literal >}withExceptionHandler(myDefinedThrowableExceptionHandler)
     *         .{@literal <}RuntimeException{@literal >}withExceptionHandler(myDefinedRuntimeExceptionExceptionHandler);
     * </code>
     *
     * And a {@link RuntimeException} is thrown, then the myDefinedThrowableExceptionHandler would catch it and handle it.
     *
     * Additionally, any non handled exceptions, unless ignored using {@link DefaultExecutionFactoryBuilder#withIgnoredExceptionType(Class)},
     * will be wrapped in an {@link UnhandledException} and thrown as such.
     *
     * @param exceptionHandler The handler for the exception.
     * @param <T>              The subtype of {@link Throwable} that will be handled here.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    public <T extends Throwable> SELF withExceptionHandler(DefinedExceptionHandler<T> exceptionHandler) {
        return addToList(exceptionHandlers, Stream.of(exceptionHandler));
    }

    /**
     * Allows a specific {@link RuntimeException} and it's subclasses to be ignored by the exception handlers.
     *
     * @param exceptionClass The class of the runtime exception to ignore. it must extend {@link RuntimeException}
     * @param <T>            The type of the {@link RuntimeException} to ignore.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    public <T extends RuntimeException> SELF withIgnoredExceptionType(Class<T> exceptionClass) {
        Optional.ofNullable(exceptionClass).map(PassThroughExceptionHandler::new).ifPresent(handler -> exceptionHandlers.add(0, handler));
        return (SELF) this;
    }

    private <T> SELF addToList(List<T> list, Stream<T> elementsToAdd) {
        elementsToAdd.forEach(list::add);
        return (SELF) this;
    }

    /**
     * Builds the {@link ExecutionFactory}.
     *
     * @return Execution The execution context built based on this builder.
     */
    public ExecutionFactory buildExecutionFactory() {
        return new ExecutionFactory(() -> onPreExecutionListeners.stream().forEach(OnPreExecutionListener::onPreExecution),
                result -> onSuccessListeners.stream().forEach(onSuccessListener -> onSuccessListener.onSuccess(result)),
                throwable -> exceptionHandlers.stream()
                        .filter(handler -> handler.getHandledException().isInstance(throwable))
                        .findFirst()
                        .orElseThrow(() -> new UnhandledException(throwable))
                        .handle(throwable));
    }
}
