package org.mule.commons.atlantic.execution.builder.factory;

import org.mule.commons.atlantic.execution.builder.lambda.runnable.InstanceRunnableBuilderFactory;
import org.mule.commons.atlantic.execution.builder.lambda.supplier.InstanceSupplierBuilderFactory;
import org.mule.commons.atlantic.lambda.supplier.Supplier;

/**
 * DefaultExecutionFactoryBuilder in charge of executing instance methods.
 *
 * This class contains a set of configuration methods an a series overloaded methods called execute.
 * Each of these execute methods allow for the execution of a different type of method, starting from instance methods
 * of no parameters to up to 10 parameters on it.
 * The configuration methods allow to set a list of pre execution listeners, another list of post execution listeners,
 * a list of exception handlers and the handler for the thread execution.
 *
 * The execution for such methods will follow the following code:
 * <code>
 *     MyResult result = new InstanceExecutionFactoryBuilder(myInstance).execute(MyInstance::myMethod)
 *         .withParam(param1)
 *         .withParam(param2);
 * </code>
 *
 * Where myInstance is an instance of the MyInstance class, myMethod is an instance method on it, param1 is the first
 * parameter on myMethod, param2 is the second (and last) parameter on myMethod and MyResult is the returning object of
 * myMethod. This means that that execution is the equivalent of doing:
 *
 * <code>
 *     MyResult result = myInstance.myMethod(param1, param2);
 * </code>
 *
 * The advantage of doing this in this way is all the wrappers and listeners that we can add to the execution.
 *
 * @param <INSTANCE> The type of object over which the execution will be made. This can be an interface.
 * */
public class InstanceExecutionFactoryBuilder<INSTANCE> extends DefaultExecutionFactoryBuilder<InstanceExecutionFactoryBuilder<INSTANCE>> implements InstanceSupplierBuilderFactory<INSTANCE, InstanceExecutionFactoryBuilder<INSTANCE>>, InstanceRunnableBuilderFactory<INSTANCE, InstanceExecutionFactoryBuilder<INSTANCE>> {

    private final Supplier<INSTANCE> instanceSupplier;

    /**
     * Default constructor. Takes the instance over which the execution will be made.
     *
     * @param instanceSupplier The supplier for the instance over which the execution will be made.
     */
    public InstanceExecutionFactoryBuilder(Supplier<INSTANCE> instanceSupplier) {
        this.instanceSupplier = instanceSupplier;
    }

    @Override
    public Supplier<INSTANCE> getInstanceSupplier() {
        return instanceSupplier;
    }
}
