package org.mule.commons.atlantic.lambda.function;

import org.mule.commons.atlantic.lambda.supplier.Supplier;

import java.util.Objects;


/**
 * Represents a function that accepts six arguments and produces a result.
 * This is the six-arity specialization of {@link Function}.
 *
 * This is a functional interface whose functional method is {@link #apply(Object, Object, Object, Object, Object, Object)}.
 *
 * @param <A>      The type of the first argument to the function
 * @param <B>      The type of the second argument to the function
 * @param <C>      The type of the third argument to the function
 * @param <D>      The type of the fourth argument to the function
 * @param <E>      The type of the fifth argument to the function
 * @param <F>      The type of the sixth argument to the function
 * @param <RESULT> The type of the result of the function
 */
@FunctionalInterface
public interface HexaFunction<A, B, C, D, E, F, RESULT> extends AtlanticFunction<A, PentaFunction<B, C, D, E, F, RESULT>> {

    /**
     * Applies this function to the given arguments.
     *
     * @param a The first function argument.
     * @param b The second function argument.
     * @param c The third function argument.
     * @param d The fourth function argument.
     * @param e The fifth function argument.
     * @param f The sixth function argument.
     * @return Object The function result.
     * @throws Throwable Any exception that the operation will throw.
     */
    RESULT apply(A a, B b, C c, D d, E e, F f) throws Throwable;

    @Override
    default PentaFunction<B, C, D, E, F, RESULT> downgrade(Supplier<A> supplier) {
        return (b, c, d, e, f) -> apply(supplier.get(), b, c, d, e, f);
    }

    /**
     * Returns a composed hexafunction that first applies this hexafunction to its input, and then applies the {@code after}
     * function to the result.
     * If evaluation of either function throws an exception, it is relayed to the caller of the composed function.
     *
     * @param <AFTER_RESULT> The type of output of the {@code after} function, and of the composed function.
     * @param after          The function to apply after this function is applied.
     * @return HexaFunction A composed hexafunction that first applies this function and then applies the {@code after} function
     */
    default <AFTER_RESULT> HexaFunction<A, B, C, D, E, F, AFTER_RESULT> andThen(Function<? super RESULT, ? extends AFTER_RESULT> after) {
        Objects.requireNonNull(after);
        return (A a, B b, C c, D d, E e, F f) -> after.apply(apply(a, b, c, d, e, f));
    }
}
