package org.mule.commons.atlantic.lambda.function;

import org.mule.commons.atlantic.lambda.supplier.Supplier;

import java.util.Objects;


/**
 * Represents a function that accepts three arguments and produces a result.
 * This is the three-arity specialization of {@link Function}.
 *
 * This is a functional interface whose functional method is {@link #apply(Object, Object, Object)}.
 *
 * @param <A>      The type of the first argument to the function
 * @param <B>      The type of the second argument to the function
 * @param <C>      The type of the third argument to the function
 * @param <RESULT> The type of the result of the function
 */
@FunctionalInterface
public interface TriFunction<A, B, C, RESULT> extends AtlanticFunction<A, BiFunction<B, C, RESULT>> {

    /**
     * Applies this function to the given arguments.
     *
     * @param a The first function argument.
     * @param b The second function argument.
     * @param c The third function argument.
     * @return Object The function result.
     * @throws Throwable Any exception that the operation will throw.
     */
    RESULT apply(A a, B b, C c) throws Throwable;

    @Override
    default BiFunction<B, C, RESULT> downgrade(Supplier<A> supplier) {
        return (b, c) -> apply(supplier.get(), b, c);
    }

    /**
     * Returns a composed trifunction that first applies this trifunction to its input, and then applies the {@code after}
     * function to the result.
     * If evaluation of either function throws an exception, it is relayed to the caller of the composed function.
     *
     * @param <AFTER_RESULT> The type of output of the {@code after} function, and of the composed function.
     * @param after          The function to apply after this function is applied.
     * @return TriFunction A composed trifunction that first applies this function and then applies the {@code after} function
     */
    default <AFTER_RESULT> TriFunction<A, B, C, AFTER_RESULT> andThen(Function<? super RESULT, ? extends AFTER_RESULT> after) {
        Objects.requireNonNull(after);
        return (A a, B b, C c) -> after.apply(apply(a, b, c));
    }
}
