package org.mule.commons.atlantic.execution.builder.factory;

import org.mule.commons.atlantic.exception.UnhandledException;
import org.mule.commons.atlantic.execution.ExecutionFactory;
import org.mule.commons.atlantic.execution.exception.handler.DefinedExceptionHandler;
import org.mule.commons.atlantic.execution.exception.handler.ExceptionHandler;
import org.mule.commons.atlantic.execution.listener.OnPreExecutionListener;
import org.mule.commons.atlantic.execution.listener.OnSuccessListener;

import java.util.List;

public interface ExecutionFactoryBuilder<SELF extends ExecutionFactoryBuilder> {
    /**
     * Adds an array of {@link OnPreExecutionListener}s to be executed before the execution.
     *
     * @param onPreExecutionListeners The listeners to set.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    SELF withPreExecutionListener(OnPreExecutionListener... onPreExecutionListeners);

    /**
     * Adds a list of {@link OnPreExecutionListener}s to be executed before the execution.
     *
     * @param onPreExecutionListeners The listeners to set.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    SELF withPreExecutionListener(List<OnPreExecutionListener> onPreExecutionListeners);

    /**
     * Adds an array of {@link OnPreExecutionListener}s to be executed after the execution.
     *
     * @param onSuccessListeners The listeners to set.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    SELF withPostExecutionListener(OnSuccessListener... onSuccessListeners);

    /**
     * Adds a list of {@link OnPreExecutionListener}s to be executed after the execution.
     *
     * @param onSuccessListeners The listeners to set.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    SELF withPostExecutionListener(List<OnSuccessListener> onSuccessListeners);

    /**
     * Adds an {@link ExceptionHandler} to handle a specific kind of {@link Throwable} or it's subclasses. Handlers are
     * triggered in the order they were set into the context with only the first match found triggering. This means that
     * if the following case would happen:
     *
     * <code>
     *     executor.withExceptionHandler(Throwable.class, myThrowableExceptionHandler)
     *         .withExceptionHandler(RuntimeException.class, myRuntimeExceptionExceptionHandler);
     * </code>
     *
     * And a {@link RuntimeException} is thrown, then the myThrowableExceptionHandler would catch it and handle it.
     *
     * Additionally, any non handled exceptions, unless ignored using {@link DefaultExecutionFactoryBuilder#withIgnoredExceptionType(Class)},
     * will be wrapped in an {@link UnhandledException} and thrown as such.
     *
     * @param exceptionClass   The exception type to handle. This will include the specific exception and all it's
     *                         subclasses.
     * @param exceptionHandler The handler for the exception.
     * @param <T>              The subtype of {@link Throwable} that will be handled here.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    <T extends Throwable> SELF withExceptionHandler(Class<T> exceptionClass, ExceptionHandler<T> exceptionHandler);

    /**
     * Adds a {@link DefinedExceptionHandler} to handle a specific kind of {@link Throwable} or it's subclasses. Handlers are
     * triggered in the order they were set into the context with only the first match found triggering. This means that
     * if the following case would happen:
     *
     * <code>
     *     executor.{@literal <}Throwable{@literal >}withExceptionHandler(myDefinedThrowableExceptionHandler)
     *         .{@literal <}RuntimeException{@literal >}withExceptionHandler(myDefinedRuntimeExceptionExceptionHandler);
     * </code>
     *
     * And a {@link RuntimeException} is thrown, then the myDefinedThrowableExceptionHandler would catch it and handle it.
     *
     * Additionally, any non handled exceptions, unless ignored using {@link DefaultExecutionFactoryBuilder#withIgnoredExceptionType(Class)},
     * will be wrapped in an {@link UnhandledException} and thrown as such.
     *
     * @param exceptionHandler The handler for the exception.
     * @param <T>              The subtype of {@link Throwable} that will be handled here.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    <T extends Throwable> SELF withExceptionHandler(DefinedExceptionHandler<T> exceptionHandler);

    /**
     * Allows a specific {@link RuntimeException} and it's subclasses to be ignored by the exception handlers.
     *
     * @param exceptionClass The class of the runtime exception to ignore. it must extend {@link RuntimeException}
     * @param <T>            The type of the {@link RuntimeException} to ignore.
     * @return DefaultExecutionFactoryBuilder This execution builder factory.
     */
    <T extends RuntimeException> SELF withIgnoredExceptionType(Class<T> exceptionClass);

    /**
     * Builds the {@link ExecutionFactory}.
     *
     * @return Execution The execution context built based on this builder.
     */
    ExecutionFactory buildExecutionFactory();
}
