package org.mule.commons.atlantic.lambda.consumer;

import org.mule.commons.atlantic.lambda.function.TetraFunction;
import org.mule.commons.atlantic.lambda.supplier.Supplier;

/**
 * Represents an operation that accepts four input arguments and returns no result.  This is the four-arity
 * specialization of {@link Consumer}.
 * Unlike most other functional interfaces, {@code TetraConsumer} is expected to operate via side-effects.
 *
 * This is a functional interface whose functional method is {@link #accept(Object, Object, Object, Object)}.
 *
 * @param <A> The type of the first argument to the operation.
 * @param <B> The type of the second argument to the operation.
 * @param <C> The type of the third argument to the operation.
 * @param <D> The type of the fourth argument to the operation.
 */
@FunctionalInterface
public interface TetraConsumer<A, B, C, D> extends AtlanticConsumer<A, TriConsumer<B, C, D>, TetraFunction<A, B, C, D, Void>> {

    @Override
    default TriConsumer<B, C, D> downgrade(Supplier<A> supplier) {
        return (b, c, d) -> accept(supplier.get(), b, c, d);
    }

    /**
     * Performs this operation on the given arguments.
     *
     * @param a The first input argument.
     * @param b The second input argument.
     * @param c The third input argument.
     * @param d The fourth input argument.
     * @throws Throwable Any exception that the operation will throw.
     */
    void accept(A a, B b, C c, D d) throws Throwable;

    @Override
    default TetraFunction<A, B, C, D, Void> toFunction() {
        return (a, b, c, d) -> {
            accept(a, b, c, d);
            return null;
        };
    }

}
