package org.mule.commons.atlantic.lambda.supplier;

import org.mule.commons.atlantic.lambda.AtlanticLambda;
import org.mule.commons.atlantic.lambda.function.Function;

import java.util.Objects;

/**
 * Represents a supplier of results.
 *
 * There is no requirement that a new or distinct result be returned each time the supplier is invoked.
 *
 * This is a functional interface whose functional method is {@link #get()}.
 *
 * @param <RESULT> the type of results supplied by this supplier
 */
@FunctionalInterface
public interface Supplier<RESULT> extends AtlanticLambda {

    /**
     * Gets a result.
     *
     * @return Object The result.
     * @throws Throwable Any exception that the operation will throw.
     */
    RESULT get() throws Throwable;

    static <RESULT> Supplier<RESULT> fixedSupplier(RESULT result) {
        return () -> result;
    }

    /**
     * Returns a composed function that first applies this function to its input, and then applies the {@code after}
     * function to the result.
     * If evaluation of either function throws an exception, it is relayed to the caller of the composed function.
     *
     * @param <AFTER_RESULT> The type of output of the {@code after} function, and of the composed function.
     * @param after          The function to apply after this function is applied.
     * @return Function A composed function that first applies this function and then applies the {@code after} function
     */
    default <AFTER_RESULT> Supplier<AFTER_RESULT> andThen(Function<? super RESULT, ? extends AFTER_RESULT> after) {
        Objects.requireNonNull(after);
        return () -> after.apply(get());
    }

}
