/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.oauth.internal;

import static java.lang.String.format;
import static java.lang.String.valueOf;
import static java.lang.Thread.currentThread;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.MediaType.ANY;
import static org.mule.runtime.api.metadata.MediaType.parse;
import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.BAD_REQUEST;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.INTERNAL_SERVER_ERROR;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.MOVED_TEMPORARILY;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpConstants.Method.GET;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Names.LOCATION;
import static org.mule.runtime.http.api.utils.HttpEncoderDecoderUtils.appendQueryParam;
import static org.mule.runtime.oauth.api.OAuthAuthorizationStatusCode.AUTHORIZATION_CODE_RECEIVED_STATUS;
import static org.mule.runtime.oauth.api.OAuthAuthorizationStatusCode.AUTHORIZATION_STATUS_QUERY_PARAM_KEY;
import static org.mule.runtime.oauth.api.OAuthAuthorizationStatusCode.NO_AUTHORIZATION_CODE_STATUS;
import static org.mule.runtime.oauth.api.OAuthAuthorizationStatusCode.TOKEN_NOT_FOUND_STATUS;
import static org.mule.runtime.oauth.api.OAuthAuthorizationStatusCode.TOKEN_URL_CALL_FAILED_STATUS;
import static org.mule.runtime.oauth.api.state.ResourceOwnerOAuthContext.DEFAULT_RESOURCE_OWNER_ID;
import static org.mule.runtime.oauth.internal.OAuthConstants.CODE_PARAMETER;
import static org.mule.runtime.oauth.internal.OAuthConstants.GRANT_TYPE_AUTHENTICATION_CODE;
import static org.mule.runtime.oauth.internal.OAuthConstants.GRANT_TYPE_PARAMETER;
import static org.mule.runtime.oauth.internal.OAuthConstants.GRANT_TYPE_REFRESH_TOKEN;
import static org.mule.runtime.oauth.internal.OAuthConstants.REDIRECT_URI_PARAMETER;
import static org.mule.runtime.oauth.internal.OAuthConstants.REFRESH_TOKEN_PARAMETER;
import static org.mule.runtime.oauth.internal.OAuthConstants.STATE_PARAMETER;
import static org.mule.runtime.oauth.internal.util.ClassLoaderUtils.setContextClassLoader;
import static org.slf4j.LoggerFactory.getLogger;

import org.apache.commons.lang3.StringUtils;
import org.mule.runtime.api.exception.DefaultMuleException;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Lifecycle;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.HttpConstants.HttpStatus;
import org.mule.runtime.http.api.HttpConstants.Method;
import org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;
import org.mule.runtime.http.api.domain.request.HttpRequestContext;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.runtime.http.api.server.RequestHandlerManager;
import org.mule.runtime.http.api.server.async.HttpResponseReadyCallback;
import org.mule.runtime.http.api.server.async.ResponseStatusCallback;
import org.mule.runtime.oauth.api.AuthorizationCodeOAuthDancer;
import org.mule.runtime.oauth.api.builder.AuthorizationCodeDanceCallbackContext;
import org.mule.runtime.oauth.api.exception.RequestAuthenticationException;
import org.mule.runtime.oauth.api.exception.TokenNotFoundException;
import org.mule.runtime.oauth.api.exception.TokenUrlResponseException;
import org.mule.runtime.oauth.api.listener.AuthorizationCodeListener;
import org.mule.runtime.oauth.api.state.ResourceOwnerOAuthContextWithRefreshState;
import org.mule.runtime.oauth.internal.authorizationcode.AuthorizationRequestUrlBuilder;
import org.mule.runtime.oauth.internal.authorizationcode.DefaultAuthorizationCodeRequest;
import org.mule.runtime.oauth.internal.config.DefaultAuthorizationCodeOAuthDancerConfig;
import org.mule.runtime.oauth.internal.state.StateDecoder;
import org.mule.runtime.oauth.internal.state.StateEncoder;
import org.mule.runtime.oauth.internal.state.TokenResponse;
import org.mule.runtime.oauth.internal.util.IOUtils;
import org.slf4j.Logger;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

/**
 * Provides OAuth dance support for authorization-code grant-type.
 *
 * @since 1.0
 */
public class DefaultAuthorizationCodeOAuthDancer extends AbstractOAuthDancer<DefaultAuthorizationCodeOAuthDancerConfig>
    implements AuthorizationCodeOAuthDancer, Lifecycle {

  private static final Logger LOGGER = getLogger(DefaultAuthorizationCodeOAuthDancer.class);
  private RequestHandlerManager redirectUrlHandlerManager;
  private RequestHandlerManager localAuthorizationUrlHandlerManager;

  public DefaultAuthorizationCodeOAuthDancer(DefaultAuthorizationCodeOAuthDancerConfig config) {
    super(config);
  }

  @Override
  public void initialise() throws InitialisationException {
    config.getHttpServer().ifPresent(s -> {
      redirectUrlHandlerManager = addRequestHandler(s, GET, config.getLocalCallbackUrlPath(), createRedirectUrlListener());
      localAuthorizationUrlHandlerManager =
          addRequestHandler(s, GET, config.getLocalAuthorizationUrlPath(), createLocalAuthorizationUrlListener());
    });
  }

  @Override
  public void addListener(AuthorizationCodeListener listener) {
    doAddListener(listener);
  }

  @Override
  public void removeListener(AuthorizationCodeListener listener) {
    doRemoveListener(listener);
  }

  private static RequestHandlerManager addRequestHandler(HttpServer server, Method method, String path,
                                                         RequestHandler callbackHandler) {
    ClassLoader appRegionClassLoader = currentThread().getContextClassLoader();

    RequestHandler requestHandler = new RequestHandler() {

      @Override
      public void handleRequest(HttpRequestContext requestContext, HttpResponseReadyCallback responseCallback) {
        final Thread thread = currentThread();
        final ClassLoader currentClassLoader = thread.getContextClassLoader();
        final ClassLoader contextClassLoader = DefaultAuthorizationCodeOAuthDancer.class.getClassLoader();
        setContextClassLoader(thread, currentClassLoader, contextClassLoader);
        try {
          callbackHandler.handleRequest(requestContext, responseCallback);
        } catch (Exception e) {
          LOGGER.error("Uncaught Exception on OAuth listener", e);
          sendErrorResponse(INTERNAL_SERVER_ERROR, e.getMessage(), responseCallback);
        } finally {
          setContextClassLoader(thread, contextClassLoader, currentClassLoader);
        }
      }

      @Override
      public ClassLoader getContextClassLoader() {
        return appRegionClassLoader;
      }
    };

    return server.addRequestHandler(singleton(method.name()), path, requestHandler);
  }

  private static void sendErrorResponse(final HttpStatus status, String message,
                                        HttpResponseReadyCallback responseCallback) {
    responseCallback.responseReady(HttpResponse.builder()
        .statusCode(status.getStatusCode())
        .reasonPhrase(status.getReasonPhrase())
        .entity(
                message != null ? new ByteArrayHttpEntity(message.getBytes()) : new EmptyHttpEntity())
        .addHeader(CONTENT_LENGTH, message != null ? valueOf(message.length()) : "0")
        .build(), new ResponseStatusCallback() {

          @Override
          public void responseSendFailure(Throwable exception) {
            LOGGER.warn("Error while sending {} response {}", status.getStatusCode(), exception.getMessage());
            if (LOGGER.isDebugEnabled()) {
              LOGGER.debug("Exception thrown", exception);
            }
          }

          @Override
          public void responseSendSuccessfully() {}
        });
  }

  private RequestHandler createRedirectUrlListener() {
    ClassLoader appRegionClassLoader = currentThread().getContextClassLoader();

    return new RequestHandler() {

      @Override
      public void handleRequest(HttpRequestContext requestContext, HttpResponseReadyCallback responseCallback) {
        final HttpRequest request = requestContext.getRequest();
        final MultiMap<String, String> queryParams = request.getQueryParams();

        final String state = queryParams.get(STATE_PARAMETER);
        final StateDecoder stateDecoder = new StateDecoder(state);
        final String authorizationCode = queryParams.get(CODE_PARAMETER);

        String resourceOwnerId = stateDecoder.decodeResourceOwnerId();

        if (authorizationCode == null) {
          LOGGER.info("HTTP Request to redirect URL done by the OAuth provider does not contains a code query parameter. "
              + "Code query parameter is required to get the access token.");
          LOGGER.error("Could not extract authorization code from OAuth provider HTTP request done to the redirect URL");

          sendResponse(stateDecoder, responseCallback, BAD_REQUEST,
                       "Failure retrieving access token.\n OAuth Server uri from callback: " + request.getUri(),
                       NO_AUTHORIZATION_CODE_STATUS);
          return;
        }

        AuthorizationCodeDanceCallbackContext beforeCallbackContext = config.getBeforeDanceCallback()
            .apply(new DefaultAuthorizationCodeRequest(resourceOwnerId, config.getAuthorizationUrl(), config.getTokenUrl(),
                                                       config.getCredentialConfig().getClientId(),
                                                       config.getCredentialConfig().getClientSecret(),
                                                       config.getScopes(),
                                                       stateDecoder.decodeOriginalState()));

        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Redirect url request state: " + state);
          LOGGER.debug("Redirect url request code: " + authorizationCode);
        }

        final Map<String, String> formData = new HashMap<>();
        formData.put(CODE_PARAMETER, authorizationCode);
        String authorization = handleClientCredentials(formData);
        formData.put(GRANT_TYPE_PARAMETER, GRANT_TYPE_AUTHENTICATION_CODE);
        formData.put(REDIRECT_URI_PARAMETER, config.getExternalCallbackUrl());

        invokeTokenUrl(config.getTokenUrl(), formData, emptyMultiMap(), emptyMultiMap(), authorization, true,
                       config.getEncoding())
                           .exceptionally(e -> {
                             final Thread thread = currentThread();
                             final ClassLoader currentClassLoader = thread.getContextClassLoader();
                             final ClassLoader contextClassLoader = DefaultAuthorizationCodeOAuthDancer.class.getClassLoader();
                             setContextClassLoader(thread, currentClassLoader, contextClassLoader);
                             try {
                               if (e.getCause() instanceof TokenUrlResponseException) {
                                 LOGGER.error(e.getMessage());
                                 sendResponse(stateDecoder, responseCallback, INTERNAL_SERVER_ERROR,
                                              format("Failure calling token url %s. Exception message is %s",
                                                     config.getTokenUrl(), e.getMessage()),
                                              TOKEN_URL_CALL_FAILED_STATUS);

                               } else if (e.getCause() instanceof TokenNotFoundException) {
                                 LOGGER.error(e.getMessage());
                                 sendResponse(stateDecoder, responseCallback, INTERNAL_SERVER_ERROR,
                                              "Failed getting access token or refresh token from token URL response. See logs for details.",
                                              TOKEN_NOT_FOUND_STATUS);

                               } else {
                                 LOGGER.error("Uncaught Exception on OAuth listener", e);
                                 sendErrorResponse(INTERNAL_SERVER_ERROR, e.getMessage(), responseCallback);
                               }
                             } finally {
                               setContextClassLoader(thread, contextClassLoader, currentClassLoader);
                             }
                             return null;
                           }).thenAccept(tokenResponse -> {
                             Thread thread = currentThread();
                             final ClassLoader currentClassLoader = thread.getContextClassLoader();
                             final ClassLoader contextClassLoader = DefaultAuthorizationCodeOAuthDancer.class.getClassLoader();
                             setContextClassLoader(thread, currentClassLoader, contextClassLoader);
                             try {
                               if (tokenResponse == null) {
                                 // This is just for the case where an error was already handled
                                 return;
                               }

                               final ResourceOwnerOAuthContextWithRefreshState resourceOwnerOAuthContext =
                                   (ResourceOwnerOAuthContextWithRefreshState) getContextForResourceOwner(resourceOwnerId == null
                                       ? DEFAULT_RESOURCE_OWNER_ID
                                       : resourceOwnerId);

                               if (LOGGER.isDebugEnabled()) {
                                 LOGGER.debug("Update OAuth Context for resourceOwnerId %s",
                                              resourceOwnerOAuthContext.getResourceOwnerId());
                                 LOGGER.debug("Retrieved access token, refresh token and expires from token url are: %s, %s, %s",
                                              tokenResponse.getAccessToken(), tokenResponse.getRefreshToken(),
                                              tokenResponse.getExpiresIn());
                               }

                               updateResourceOwnerState(resourceOwnerOAuthContext, stateDecoder.decodeOriginalState(),
                                                        tokenResponse);
                               updateResourceOwnerOAuthContext(resourceOwnerOAuthContext);

                               forEachListener(l -> l.onAuthorizationCompleted(resourceOwnerOAuthContext));
                               config.getAfterDanceCallback().accept(beforeCallbackContext, resourceOwnerOAuthContext);

                               sendResponse(stateDecoder, responseCallback, OK, "Successfully retrieved access token",
                                            AUTHORIZATION_CODE_RECEIVED_STATUS);
                             } finally {
                               setContextClassLoader(thread, contextClassLoader, currentClassLoader);
                             }
                           });
      }

      @Override
      public ClassLoader getContextClassLoader() {
        return appRegionClassLoader;
      }
    };
  }

  private void sendResponse(StateDecoder stateDecoder, HttpResponseReadyCallback responseCallback,
                            HttpStatus statusEmptyState, String message, int authorizationStatus) {
    String onCompleteRedirectToValue = stateDecoder.decodeOnCompleteRedirectTo();
    if (!isEmpty(onCompleteRedirectToValue)) {
      sendResponse(responseCallback, MOVED_TEMPORARILY, message, appendQueryParam(onCompleteRedirectToValue,
                                                                                  AUTHORIZATION_STATUS_QUERY_PARAM_KEY,
                                                                                  valueOf(authorizationStatus)));
    } else {
      sendResponse(responseCallback, statusEmptyState, message);
    }
  }

  private void sendResponse(HttpResponseReadyCallback responseCallback, HttpStatus status, String message,
                            String locationHeader) {
    HttpResponseBuilder httpResponseBuilder = HttpResponse.builder();
    httpResponseBuilder.statusCode(status.getStatusCode());
    httpResponseBuilder.reasonPhrase(status.getReasonPhrase());
    httpResponseBuilder.entity(new ByteArrayHttpEntity(message.getBytes()));
    httpResponseBuilder.addHeader(CONTENT_LENGTH, valueOf(message.length()));
    httpResponseBuilder.addHeader(LOCATION, locationHeader);
    httpResponseBuilder.headers(new MultiMap<>(config.getCustomHeaders().get()));
    responseCallback.responseReady(httpResponseBuilder.build(), new ResponseStatusCallback() {

      @Override
      public void responseSendFailure(Throwable exception) {
        LOGGER.warn("Error while sending {} response {}", status.getStatusCode(), exception.getMessage());
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Exception thrown", exception);
        }
      }

      @Override
      public void responseSendSuccessfully() {}
    });
  }

  private static void sendResponse(HttpResponseReadyCallback responseCallback, HttpStatus status, String message) {
    HttpResponseBuilder httpResponseBuilder = HttpResponse.builder();
    httpResponseBuilder.statusCode(status.getStatusCode());
    httpResponseBuilder.reasonPhrase(status.getReasonPhrase());
    httpResponseBuilder.entity(new ByteArrayHttpEntity(message.getBytes()));
    httpResponseBuilder.addHeader(CONTENT_LENGTH, valueOf(message.length()));
    responseCallback.responseReady(httpResponseBuilder.build(), new ResponseStatusCallback() {

      @Override
      public void responseSendFailure(Throwable exception) {
        LOGGER.warn("Error while sending {} response {}", status.getStatusCode(), exception.getMessage());
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Exception thrown", exception);
        }
      }

      @Override
      public void responseSendSuccessfully() {}
    });
  }

  private static boolean isEmpty(String value) {
    return value == null || StringUtils.isEmpty(value) || "null".equals(value);
  }

  private RequestHandler createLocalAuthorizationUrlListener() {
    ClassLoader appRegionClassLoader = currentThread().getContextClassLoader();

    return new RequestHandler() {

      @Override
      public void handleRequest(HttpRequestContext requestContext, HttpResponseReadyCallback responseCallback) {
        handleLocalAuthorizationRequest(requestContext.getRequest(), responseCallback);
      }

      @Override
      public ClassLoader getContextClassLoader() {
        return appRegionClassLoader;
      }
    };
  }

  @Override
  public void handleLocalAuthorizationRequest(HttpRequest request, HttpResponseReadyCallback responseCallback) {
    final String body = readBody(request);
    final MultiMap<String, String> headers = readHeaders(request);
    final MediaType mediaType = getMediaType(request);
    final MultiMap<String, String> queryParams = request.getQueryParams();

    final String originalState = resolveExpression(config.getState(), body, headers, queryParams, mediaType);
    final StateEncoder stateEncoder = new StateEncoder(originalState);

    final String resourceOwnerId =
        resolveExpression(config.getLocalAuthorizationUrlResourceOwnerId(), body, headers, queryParams, mediaType);
    if (resourceOwnerId != null) {
      stateEncoder.encodeResourceOwnerIdInState(resourceOwnerId);
    }

    final String onCompleteRedirectToValue = queryParams.get("onCompleteRedirectTo");
    if (onCompleteRedirectToValue != null) {
      stateEncoder.encodeOnCompleteRedirectToInState(onCompleteRedirectToValue);
    }

    final String authorizationUrlWithParams = new AuthorizationRequestUrlBuilder()
        .setAuthorizationUrl(config.getAuthorizationUrl())
        .setClientId(config.getCredentialConfig().getClientId())
        .setClientSecret(config.getCredentialConfig().getClientSecret())
        .setCustomParameters(config.getCustomParameters().get())
        .setRedirectUrl(config.getExternalCallbackUrl())
        .setState(stateEncoder.getEncodedState())
        .setScope(config.getScopes())
        .setEncoding(config.getEncoding())
        .buildUrl();

    sendResponse(responseCallback, MOVED_TEMPORARILY, body, authorizationUrlWithParams);
  }

  private String readBody(final HttpRequest request) {
    return IOUtils.toString(request.getEntity().getContent());
  }

  private MultiMap<String, String> readHeaders(final HttpRequest request) {
    return request.getHeaders();
  }

  private MediaType getMediaType(final HttpRequest request) {
    String contentType = request.getHeaderValue(CONTENT_TYPE);
    return contentType != null ? parse(contentType) : ANY;
  }

  @Override
  public void start() throws MuleException {
    super.start();
    if (config.getHttpServer().isPresent()) {
      try {
        config.getHttpServer().get().start();
      } catch (IOException e) {
        throw new DefaultMuleException(e);
      }
      redirectUrlHandlerManager.start();
      localAuthorizationUrlHandlerManager.start();
    }
  }

  @Override
  public void stop() throws MuleException {
    if (config.getHttpServer().isPresent()) {
      redirectUrlHandlerManager.stop();
      localAuthorizationUrlHandlerManager.stop();
      config.getHttpServer().get().stop();
    }
    super.stop();
  }

  @Override
  public void dispose() {
    if (config.getHttpServer().isPresent()) {
      redirectUrlHandlerManager.dispose();
      localAuthorizationUrlHandlerManager.dispose();
      config.getHttpServer().get().dispose();
    }
  }

  @Override
  public CompletableFuture<String> accessToken(String resourceOwner) throws RequestAuthenticationException {
    final String accessToken = getContextForResourceOwner(resourceOwner).getAccessToken();
    if (accessToken == null) {
      throw new RequestAuthenticationException(createStaticMessage(format("No access token found. "
          + "Verify that you have authenticated before trying to execute an operation to the API.")));
    }

    // TODO MULE-11858 proactively refresh if the token has already expired based on its 'expiresIn' parameter
    return completedFuture(accessToken);
  }

  @Override
  public CompletableFuture<Void> refreshToken(String resourceOwner) {
    return refreshToken(resourceOwner, false);
  }

  @Override
  public CompletableFuture<Void> refreshToken(String resourceOwner, boolean useQueryParameters) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Executing refresh token for user " + resourceOwner);
    }

    return doRefreshToken(() -> getContextForResourceOwner(resourceOwner),
                          ctx -> doRefreshTokenRequest(useQueryParameters, (ResourceOwnerOAuthContextWithRefreshState) ctx));
  }

  protected CompletableFuture<Void> doRefreshTokenRequest(boolean useQueryParameters,
                                                          final ResourceOwnerOAuthContextWithRefreshState resourceOwnerOAuthContext) {
    final String userRefreshToken = resourceOwnerOAuthContext.getRefreshToken();
    if (userRefreshToken == null) {
      throw new MuleRuntimeException(createStaticMessage(
                                                         "The user with user id %s has no refresh token in his OAuth state so we can't execute the refresh token call",
                                                         resourceOwnerOAuthContext.getResourceOwnerId()));
    }

    final MultiMap<String, String> requestParameters = new MultiMap<>();
    requestParameters.put(REFRESH_TOKEN_PARAMETER, userRefreshToken);
    String authorization = handleClientCredentials(requestParameters);
    requestParameters.put(GRANT_TYPE_PARAMETER, GRANT_TYPE_REFRESH_TOKEN);
    requestParameters.put(REDIRECT_URI_PARAMETER, config.getExternalCallbackUrl());

    MultiMap<String, String> queryParams;
    Map<String, String> formData;

    if (useQueryParameters) {
      queryParams = requestParameters;
      formData = emptyMap();
    } else {
      queryParams = emptyMultiMap();
      formData = requestParameters;
    }

    return invokeTokenUrl(config.getTokenUrl(), formData, queryParams, emptyMultiMap(), authorization, true, config.getEncoding())
        .thenAccept(tokenResponse -> {
          final Thread thread = currentThread();
          final ClassLoader currentClassLoader = thread.getContextClassLoader();
          final ClassLoader contextClassLoader = DefaultAuthorizationCodeOAuthDancer.class.getClassLoader();
          setContextClassLoader(thread, currentClassLoader, contextClassLoader);
          try {
            if (LOGGER.isDebugEnabled()) {
              LOGGER.debug("Update OAuth Context for resourceOwnerId %s", resourceOwnerOAuthContext.getResourceOwnerId());
            }
            updateResourceOwnerState(resourceOwnerOAuthContext, null, tokenResponse);
            updateOAuthContextAfterTokenResponse(resourceOwnerOAuthContext);
            forEachListener(l -> l.onTokenRefreshed(resourceOwnerOAuthContext));
          } finally {
            setContextClassLoader(thread, contextClassLoader, currentClassLoader);
          }
        })
        .exceptionally(tokenUrlExceptionHandler(resourceOwnerOAuthContext));
  }

  private void updateResourceOwnerState(ResourceOwnerOAuthContextWithRefreshState resourceOwnerOAuthContext, String newState,
                                        TokenResponse tokenResponse) {
    resourceOwnerOAuthContext.setAccessToken(tokenResponse.getAccessToken());
    if (tokenResponse.getRefreshToken() != null) {
      resourceOwnerOAuthContext.setRefreshToken(tokenResponse.getRefreshToken());
    }
    resourceOwnerOAuthContext.setExpiresIn(tokenResponse.getExpiresIn());

    // State may be null because there's no state or because this was called after refresh token.
    if (newState != null) {
      resourceOwnerOAuthContext.setState(newState);
    }

    final Map<String, Object> customResponseParameters = tokenResponse.getCustomResponseParameters();
    for (String paramName : customResponseParameters.keySet()) {
      final Object paramValue = customResponseParameters.get(paramName);
      if (paramValue != null) {
        resourceOwnerOAuthContext.getTokenResponseParameters().put(paramName, paramValue);
      }
    }

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("New OAuth State for resourceOwnerId %s is: accessToken(%s), refreshToken(%s), expiresIn(%s), state(%s)",
                   resourceOwnerOAuthContext.getResourceOwnerId(), resourceOwnerOAuthContext.getAccessToken(),
                   isBlank(resourceOwnerOAuthContext.getRefreshToken()) ? "Not issued"
                       : resourceOwnerOAuthContext.getRefreshToken(),
                   resourceOwnerOAuthContext.getExpiresIn(), resourceOwnerOAuthContext.getState());
    }
  }

  private void forEachListener(Consumer<AuthorizationCodeListener> action) {
    onEachListener(l -> action.accept((AuthorizationCodeListener) l));
  }
}
