/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.oauth.internal.builder;

import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import org.mule.runtime.api.el.MuleExpressionLanguage;
import org.mule.runtime.api.lock.LockFactory;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.oauth.api.ClientCredentialsOAuthDancer;
import org.mule.runtime.oauth.api.builder.OAuthClientCredentialsDancerBuilder;
import org.mule.runtime.oauth.api.http.HttpClientFactory;
import org.mule.runtime.oauth.api.listener.ClientCredentialsListener;
import org.mule.runtime.oauth.api.state.ResourceOwnerOAuthContext;
import org.mule.runtime.oauth.internal.DefaultClientCredentialsOAuthDancer;
import org.mule.runtime.oauth.internal.config.AuthDancerCredentialConfig;
import org.mule.runtime.oauth.internal.config.DefaultClientCredentialsOAuthDancerConfig;
import org.mule.runtime.oauth.internal.config.TokenParserConfig;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * Builder that allows to configure the attributes for the client credentials grant type.
 *
 * @since 1.0
 */
public class DefaultOAuthClientCredentialsDancerBuilder extends AbstractOAuthDancerBuilder<ClientCredentialsOAuthDancer>
    implements OAuthClientCredentialsDancerBuilder {

  private final List<ClientCredentialsListener> listeners = new LinkedList<>();
  private final MultiMap<String, String> customParameters = new MultiMap<>();
  private final MultiMap<String, String> customHeaders = new MultiMap<>();

  public DefaultOAuthClientCredentialsDancerBuilder(SchedulerService schedulerService, LockFactory lockProvider,
                                                    Map<String, ResourceOwnerOAuthContext> tokensStore,
                                                    HttpClientFactory httpClientFactory,
                                                    MuleExpressionLanguage expressionEvaluator) {
    super(schedulerService, lockProvider, tokensStore, httpClientFactory, expressionEvaluator);
  }

  @Override
  public OAuthClientCredentialsDancerBuilder encodeClientCredentialsInBody(boolean encodeClientCredentialsInBody) {
    return (OAuthClientCredentialsDancerBuilder) super.encodeClientCredentialsInBody(encodeClientCredentialsInBody);
  }

  @Override
  public OAuthClientCredentialsDancerBuilder customParameters(Map<String, String> customParameters) {
    requireNonNull(customParameters, "customParameters cannot be null");
    putAll(customParameters, this.customParameters);

    return this;
  }

  private void putAll(Map<String, String> source, MultiMap<String, String> target) {
    if (source instanceof MultiMap) {
      target.putAll((MultiMap<String, String>) source);
    } else {
      target.putAll(source);
    }
  }

  @Override
  public OAuthClientCredentialsDancerBuilder customHeaders(Map<String, String> customHeaders) {
    requireNonNull(customHeaders, "customHeaders cannot be null");
    putAll(customHeaders, this.customHeaders);

    return this;
  }

  @Override
  public OAuthClientCredentialsDancerBuilder addListener(ClientCredentialsListener listener) {
    requireNonNull(listener, "Cannot add a null listener");
    listeners.add(listener);

    return this;
  }

  @Override
  public ClientCredentialsOAuthDancer build() {
    checkArgument(isNotBlank(clientId), "clientId cannot be blank");
    checkArgument(isNotBlank(clientSecret), "clientSecret cannot be blank");
    checkArgument(isNotBlank(tokenUrl), "tokenUrl cannot be blank");

    DefaultClientCredentialsOAuthDancerConfig config = new DefaultClientCredentialsOAuthDancerConfig();
    config.setName(name);

    config.setCredentialConfig(AuthDancerCredentialConfig.builder()
        .withClientId(clientId)
        .withClientSecret(clientSecret)
        .withClientCredentialsLocation(clientCredentialsLocation)
        .build());

    config.setTokenUrl(tokenUrl);
    config.setScopes(scopes);
    config.setEncoding(encoding);

    config.setTokenParserConfig(TokenParserConfig.builder()
        .withResponseAccessTokenExpr(responseAccessTokenExpr)
        .withResponseRefreshTokenExpr(responseRefreshTokenExpr)
        .withResponseExpiresInExpr(responseExpiresInExpr)
        .build());

    config.setCustomParametersExtractorsExprs(customParametersExtractorsExprs);
    config.setResourceOwnerIdTransformer(
                                         resourceOwnerIdTransformer);
    config.setSchedulerService(schedulerService);
    config.setLockProvider(lockProvider);
    config.setTokensStore(tokensStore);
    config.setHttpClient(httpClientFactory.get());
    config.setExpressionEvaluator(expressionEvaluator);
    config.setCustomParameters(customParameters);
    config.setCustomHeaders(customHeaders);
    config.setListeners(listeners);

    return new DefaultClientCredentialsOAuthDancer(config);
  }

}
