/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.oauth.client.internal.state;

import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import static java.util.Objects.requireNonNull;

/**
 * State parameter encoder. Allows to encode and decode an resourceOwnerId in the authentication request state parameter.
 *
 * @since 1.0
 */
public class StateEncoder {

  public static final String RESOURCE_OWNER_PARAM_NAME = ":resourceOwnerId";
  public static final String RESOURCE_OWNER_PARAM_NAME_ASSIGN = RESOURCE_OWNER_PARAM_NAME + "=";
  public static final String ON_COMPLETE_REDIRECT_TO_PARAM_NAME = ":onCompleteRedirectTo";
  public static final String ON_COMPLETE_REDIRECT_TO_PARAM_NAME_ASSIGN = ON_COMPLETE_REDIRECT_TO_PARAM_NAME + "=";
  private String state;

  public StateEncoder(String originalState) {
    this.state = originalState;
  }

  /**
   * Encodes the given resource owner ID in the current state.
   *
   * @param resourceOwnerId the resourceOwnerId to encode
   */
  public void encodeResourceOwnerIdInState(final String resourceOwnerId) {
    encodeParameter(resourceOwnerId, RESOURCE_OWNER_PARAM_NAME_ASSIGN);
  }

  /**
   * Encodes the given redirect URL to send the user when the oauth dance is complete in the current state.
   *
   * @param onCompleteRedirectToValue the redirect url to encode
   * @return an updated state with the original content plus the redirect url.
   */
  public void encodeOnCompleteRedirectToInState(String onCompleteRedirectToValue) {
    encodeParameter(onCompleteRedirectToValue, ON_COMPLETE_REDIRECT_TO_PARAM_NAME_ASSIGN);
  }

  private void encodeParameter(String parameterValue, String parameterAssignation) {
    checkArgument(parameterValue != null, "parameter cannot be null");
    checkArgument(state == null || !state.contains(ON_COMPLETE_REDIRECT_TO_PARAM_NAME_ASSIGN),
                  "parameter cannot be added after " + ON_COMPLETE_REDIRECT_TO_PARAM_NAME);
    if (state == null) {
      state = parameterAssignation + parameterValue;
    } else {
      state = state + parameterAssignation + parameterValue;
    }
  }

  public String getEncodedState() {
    return state;
  }
}
