/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.oauth.client.internal.util;

/**
 * Set of utils loading resources and classes.
 *
 * @since 1.0
 */
public class ClassLoaderUtils {

  private ClassLoaderUtils() {}

  /**
   * Sets {@code newClassLoader} as the context class loader for the {@code thread}, as long as said classloader is not the same
   * instance as {@code currentClassLoader}.
   * <p>
   * Since obtaining and setting the context classloader from a thread are expensive operations, the purpose of this method is to
   * avoid performing those operations when possible, which is why the two classloaders are tested not to be the same before
   * performing the set operation. For this method to make sense, {@code currentClassLoader} should actually be the current
   * context classloader from the {@code thread}.
   * <p>
   * This is how a typical use should look like:
   *
   * <pre>
   * Thread thread = Thread.currentThread();
   * ClassLoader currentClassLoader = thread.getContextClassLoader();
   * ClassLoader newClassLoader = getNewContextClassLoader(); // this one depends on your logic
   * ClassUtils.setContextClassLoader(thread, currentClassLoader, newClassLoader);
   * try {
   *   // execute your logic
   * } finally {
   *   // set things back as they were by reversing the arguments order
   *   ClassUtils.setContextClassLoader(thread, newClassLoader, currentClassLoader);
   * }
   * </pre>
   *
   * @param thread             the thread which context classloader is to be changed
   * @param currentClassLoader the thread's current context classloader
   * @param newClassLoader     the new classloader to be set
   */
  public static void setContextClassLoader(Thread thread, ClassLoader currentClassLoader, ClassLoader newClassLoader) {
    if (currentClassLoader != newClassLoader) {
      thread.setContextClassLoader(newClassLoader);
    }
  }
}
