/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.test.oauth.authorizationcode;

import static org.mule.runtime.test.AllureConstants.OAuthClientFeature.OAuthClientStory.AUTHORIZATION_CODE;

import static java.util.Collections.singletonList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.sameInstance;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.oauth.client.api.listener.AuthorizationCodeListener;
import org.mule.oauth.client.api.listener.OAuthStateListener;
import org.mule.oauth.client.api.state.ResourceOwnerOAuthContext;
import org.mule.oauth.client.internal.DefaultAuthorizationCodeOAuthDancer;
import org.mule.oauth.client.internal.config.DefaultAuthorizationCodeOAuthDancerConfig;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.util.List;

import io.qameta.allure.Story;
import org.junit.Test;

@Story(AUTHORIZATION_CODE)
public class DefaultAuthorizationCodeOAuthDancerTestCase extends AbstractMuleTestCase {

  @Test
  public void testListenerFromConfig() {
    DefaultAuthorizationCodeOAuthDancerConfig config = new DefaultAuthorizationCodeOAuthDancerConfig();
    AuthorizationCodeListener listener = new AuthorizationCodeListener() {};
    config.setListeners(singletonList(listener));
    TestAuthorizationCodeOAuthDancer dancer = new TestAuthorizationCodeOAuthDancer(config);

    assertThat(dancer.getListeners(), hasSize(1));
    assertThat(dancer.getListeners().get(0), sameInstance(listener));

    dancer.removeListener(listener);
    assertThat(dancer.getListeners(), hasSize(0));
  }

  @Test
  public void testAddedListener() {
    DefaultAuthorizationCodeOAuthDancerConfig config = new DefaultAuthorizationCodeOAuthDancerConfig();
    TestAuthorizationCodeOAuthDancer dancer = new TestAuthorizationCodeOAuthDancer(config);
    AuthorizationCodeListener listener = new AuthorizationCodeListener() {};
    dancer.addListener(listener);

    assertThat(dancer.getListeners(), hasSize(1));
    assertThat(dancer.getListeners().get(0), sameInstance(listener));

    dancer.removeListener(listener);
    assertThat(dancer.getListeners(), hasSize(0));
  }

  @Test
  public void testAddedListenerForResourceOwner() {
    DefaultAuthorizationCodeOAuthDancerConfig config = new DefaultAuthorizationCodeOAuthDancerConfig();
    TestAuthorizationCodeOAuthDancer dancer = new TestAuthorizationCodeOAuthDancer(config);
    AuthorizationCodeListener listener = new AuthorizationCodeListener() {};
    String resourceOwnerId = "an-owner";
    dancer.addListener(resourceOwnerId, listener);
    ResourceOwnerOAuthContext resourceOwnerOAuthContext = mock(ResourceOwnerOAuthContext.class);
    when(resourceOwnerOAuthContext.getResourceOwnerId()).thenReturn(resourceOwnerId);

    assertThat(dancer.getListeners(resourceOwnerOAuthContext), hasSize(1));
    assertThat(dancer.getListeners(resourceOwnerOAuthContext).get(0), sameInstance(listener));

    dancer.removeListener(listener);
    assertThat(dancer.getListeners(resourceOwnerOAuthContext), hasSize(1));

    dancer.removeListener(resourceOwnerId, listener);
    assertThat(dancer.getListeners(resourceOwnerOAuthContext), hasSize(0));
  }

  private static class TestAuthorizationCodeOAuthDancer extends DefaultAuthorizationCodeOAuthDancer {

    public TestAuthorizationCodeOAuthDancer(DefaultAuthorizationCodeOAuthDancerConfig config) {
      super(config);
    }

    public List<? extends OAuthStateListener> getListeners() {
      return super.getListenersToNotifyInvalidation(null);
    }

    public List<? extends OAuthStateListener> getListeners(ResourceOwnerOAuthContext resourceOwnerOAuthContext) {
      return super.getListenersToNotifyInvalidation(resourceOwnerOAuthContext);
    }

  }

}
