/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.test.oauth;

import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.test.AllureConstants.OAuthClientFeature.OAUTH_CLIENT;

import static java.lang.Thread.currentThread;
import static java.lang.Thread.sleep;
import static java.util.concurrent.Executors.newSingleThreadExecutor;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.oauth.client.api.builder.OAuthAuthorizationCodeDancerBuilder;
import org.mule.oauth.client.api.builder.OAuthClientCredentialsDancerBuilder;
import org.mule.oauth.client.api.builder.OAuthDancerBuilder;
import org.mule.oauth.client.api.state.ResourceOwnerOAuthContext;
import org.mule.oauth.client.internal.builder.DefaultOAuthAuthorizationCodeDancerBuilder;
import org.mule.oauth.client.internal.builder.DefaultOAuthClientCredentialsDancerBuilder;
import org.mule.runtime.api.el.MuleExpressionLanguage;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lock.LockFactory;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientFactory;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerFactory;
import org.mule.tck.SimpleUnitTestSupportSchedulerService;
import org.mule.tck.junit4.AbstractMuleContextTestCase;

import java.io.StringReader;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;

import org.apache.commons.io.input.ReaderInputStream;

import org.junit.After;
import org.junit.Before;

import io.qameta.allure.Feature;

import jakarta.inject.Inject;

@Feature(OAUTH_CLIENT)
public class AbstractOAuthTestCase extends AbstractMuleContextTestCase {

  protected HttpClientFactory httpClientFactory;
  protected HttpClient httpClient;
  protected HttpServer httpServer;
  protected HttpService httpService;
  protected org.mule.oauth.client.api.http.HttpClientFactory oAuthHttpClientFactory;

  protected ExecutorService httpClientCallbackExecutor;

  @Inject
  protected LockFactory lockFactory;

  public AbstractOAuthTestCase() {
    setStartContext(true);
  }

  @Override
  protected boolean doTestClassInjection() {
    return true;
  }

  @Before
  public void setupServices() throws Exception {
    httpClientCallbackExecutor = newSingleThreadExecutor();

    httpService = mock(HttpService.class);
    httpClientFactory = mock(HttpClientFactory.class);
    httpClient = mock(HttpClient.class);
    when(httpClientFactory.create(any())).thenReturn(httpClient);
    when(httpService.getClientFactory()).thenReturn(httpClientFactory);

    final HttpServerFactory httpServerFactory = mock(HttpServerFactory.class);
    httpServer = mock(HttpServer.class);
    when(httpServerFactory.create(any())).thenReturn(httpServer);
    when(httpService.getServerFactory()).thenReturn(httpServerFactory);

    oAuthHttpClientFactory = org.mule.oauth.client.api.http.HttpClientFactory.getDefault(httpService);

    final HttpResponse httpResponse = mock(HttpResponse.class);
    final InputStreamHttpEntity httpEntity = mock(InputStreamHttpEntity.class);
    when(httpEntity.getContent()).thenReturn(new ReaderInputStream(new StringReader("")));
    when(httpResponse.getEntity()).thenReturn(httpEntity);
    when(httpClient.sendAsync(any(), any())).thenAnswer(invocation -> {

      final CompletableFuture<HttpResponse> httpResponseFuture = new CompletableFuture<>();

      httpClientCallbackExecutor.execute(() -> {
        try {
          sleep(10);
        } catch (InterruptedException e) {
          currentThread().interrupt();
          httpResponseFuture.completeExceptionally(e);
        }
        httpResponseFuture.complete(httpResponse);
      });

      return httpResponseFuture;
    });
  }

  @After
  public void teardownServices() {
    httpClientCallbackExecutor.shutdown();
  }

  protected OAuthClientCredentialsDancerBuilder baseClientCredentialsDancerBuilder() {
    return baseClientCredentialsDancerBuilder(new HashMap<>());
  }

  protected OAuthClientCredentialsDancerBuilder baseClientCredentialsDancerBuilder(Map<String, ?> tokensStore) {
    DefaultOAuthClientCredentialsDancerBuilder builder =
        new DefaultOAuthClientCredentialsDancerBuilder(new SimpleUnitTestSupportSchedulerService(), lockFactory,
                                                       (Map<String, ResourceOwnerOAuthContext>) tokensStore,
                                                       oAuthHttpClientFactory,
                                                       mock(MuleExpressionLanguage.class));

    builder.clientCredentials("clientId", "clientSecret");
    return builder;
  }

  protected OAuthAuthorizationCodeDancerBuilder baseAuthCodeDancerbuilder() {

    DefaultOAuthAuthorizationCodeDancerBuilder builder =
        new DefaultOAuthAuthorizationCodeDancerBuilder(new SimpleUnitTestSupportSchedulerService(), lockFactory, new HashMap<>(),
                                                       httpService, oAuthHttpClientFactory, mock(MuleExpressionLanguage.class));

    builder.clientCredentials("clientId", "clientSecret");
    return builder;
  }

  protected <D> D startDancer(final OAuthDancerBuilder<D> builder) throws MuleException {
    final D dancer = builder.build();
    initialiseIfNeeded(dancer);
    startIfNeeded(dancer);

    return dancer;
  }
}
