/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel.builder;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.util.NamingUtil.isFriendlyName;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.PartParameter;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.ParameterDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.PartDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.SecuritySchemeDescriptor;
import org.mule.connectivity.restconnect.internal.util.NamingUtil;
import org.mule.connectivity.restconnect.internal.webapi.model.APIParameterModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APITypeModel;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.annotation.Nullable;
import javax.ws.rs.core.MediaType;

public class ConnectorParameterBuilder {

  private final ConnectorTypeDefinitionBuilder typeDefinitionBuilder;

  public ConnectorParameterBuilder(ConnectorTypeDefinitionBuilder connectorTypeDefinitionBuilder) {
    this.typeDefinitionBuilder = connectorTypeDefinitionBuilder;
  }

  public List<Parameter> buildParameterList(List<APIParameterModel> parametersModel,
                                            OperationDescriptor operationDescriptor,
                                            List<String> parameterCollisionInternalNames)
      throws ModelGenerationException {

    List<Parameter> parameterList = new ArrayList<>();
    List<String> collisionNames = new LinkedList<>(parameterCollisionInternalNames);

    for (APIParameterModel parameterModel : parametersModel) {
      ParameterDescriptor parameterDescriptor = getParameterDescriptor(parameterModel, operationDescriptor);
      if (parameterDescriptor == null || !parameterDescriptor.isIgnored()) {
        Parameter parameter = buildParameter(parameterModel, parameterDescriptor, collisionNames);
        parameterList.add(parameter);
        collisionNames.add(parameter.getInternalName());
      }
    }

    return parameterList;
  }

  public List<Parameter> buildParameterList(List<APIParameterModel> parametersModel,
                                            SecuritySchemeDescriptor securitySchemeDescriptor,
                                            List<String> parameterCollisionInternalNames)
      throws ModelGenerationException {

    List<Parameter> parameterList = new ArrayList<>();
    List<String> collisionNames = new LinkedList<>(parameterCollisionInternalNames);

    for (APIParameterModel parameterModel : parametersModel) {
      ParameterDescriptor parameterDescriptor = getParameterDescriptor(parameterModel, securitySchemeDescriptor);
      if (parameterDescriptor == null || !parameterDescriptor.isIgnored()) {
        Parameter parameter = buildParameter(parameterModel, parameterDescriptor, collisionNames);
        parameterList.add(parameter);
        collisionNames.add(parameter.getInternalName());
      }
    }

    return parameterList;
  }

  private static ParameterDescriptor getParameterDescriptor(APIParameterModel parameterModel,
                                                            SecuritySchemeDescriptor securitySchemeDescriptor) {
    if (securitySchemeDescriptor == null) {
      return null;
    }

    List<ParameterDescriptor> parameterDescriptors = new LinkedList<>();

    if (securitySchemeDescriptor.getHeaders() != null) {
      if (parameterModel.getParameterType().equals(ParameterType.HEADER)) {
        parameterDescriptors = securitySchemeDescriptor.getHeaders();
      }
      if (parameterModel.getParameterType().equals(ParameterType.QUERY)) {
        parameterDescriptors = securitySchemeDescriptor.getQueryParameters();
      }
    }

    return parameterDescriptors.stream()
        .filter(x -> isNotBlank(x.getParamName()))
        .filter(x -> x.getParamName().equalsIgnoreCase(parameterModel.getExternalName()))
        .findFirst().orElse(null);
  }

  private static ParameterDescriptor getParameterDescriptor(APIParameterModel parameterModel,
                                                            OperationDescriptor operationDescriptor) {
    if (operationDescriptor == null) {
      return null;
    }

    List<ParameterDescriptor> parameterDescriptors = new LinkedList<>();

    if (operationDescriptor.getExpects() != null) {
      if (parameterModel.getParameterType().equals(ParameterType.URI)) {
        parameterDescriptors = operationDescriptor.getExpects().getUriParameter();
      } else if (parameterModel.getParameterType().equals(ParameterType.QUERY)) {
        parameterDescriptors = operationDescriptor.getExpects().getQueryParameter();
      } else if (parameterModel.getParameterType().equals(ParameterType.HEADER)) {
        parameterDescriptors = operationDescriptor.getExpects().getHeader();
      }
    }

    return parameterDescriptors.stream()
        .filter(x -> isNotBlank(x.getParamName()))
        .filter(x -> x.getParamName().equalsIgnoreCase(parameterModel.getExternalName()))
        .findFirst().orElse(null);
  }

  public Parameter buildParameter(APIParameterModel apiParameterModel, ParameterDescriptor parameterDescriptor,
                                  List<String> collisionInternalNames)
      throws ModelGenerationException {

    return new Parameter(buildDisplayName(apiParameterModel, parameterDescriptor),
                         apiParameterModel.getExternalName(),
                         apiParameterModel.getParameterType(),
                         buildParameterTypeDefinition(apiParameterModel.getTypeModel(), null),
                         buildDescription(apiParameterModel, parameterDescriptor),
                         apiParameterModel.isRequired(),
                         apiParameterModel.getDefaultValue(),
                         apiParameterModel.isPassword(),
                         collisionInternalNames);
  }

  private static String buildDisplayName(APIParameterModel apiParameterModel, ParameterDescriptor parameterDescriptor) {
    String displayName;

    if (parameterDescriptor != null && isNotBlank(parameterDescriptor.getParamName())) {
      displayName = parameterDescriptor.getFriendlyName();
    } else {
      displayName = apiParameterModel.getDisplayName();
    }

    if (!isFriendlyName(displayName)) {
      displayName = NamingUtil.makeNameFriendly(displayName);
    }

    return displayName;
  }

  private static String buildDescription(APIParameterModel apiParameterModel, ParameterDescriptor parameterDescriptor) {
    if (parameterDescriptor != null && isNotBlank(parameterDescriptor.getParamName())) {
      return parameterDescriptor.getDescription();
    } else {
      return apiParameterModel.getDescription();
    }
  }

  private TypeDefinition buildParameterTypeDefinition(APITypeModel typeModel, @Nullable MediaType mediaType)
      throws ModelGenerationException {

    if (mediaType == null) {
      mediaType = typeModel.getMediaType() != null ? typeModel.getMediaType() : MediaType.TEXT_PLAIN_TYPE;
    }

    return typeDefinitionBuilder.buildTypeDefinition(typeModel, null, mediaType);
  }

  public PartParameter buildPartParameter(APIParameterModel apiParameterModel, @Nullable PartDescriptor partDescriptor)
      throws ModelGenerationException {
    TypeDefinition partTypeDefinition;

    MediaType mediaType = null;
    if (partDescriptor != null && isNotBlank(partDescriptor.getContentType())) {
      mediaType = MediaType.valueOf(partDescriptor.getContentType());
    }

    if (partDescriptor != null && isNotBlank(partDescriptor.getInputType())) {
      partTypeDefinition = typeDefinitionBuilder.buildTypeDefinition(partDescriptor.getInputType(), mediaType);
    } else {
      partTypeDefinition = buildParameterTypeDefinition(apiParameterModel.getTypeModel(), mediaType);
    }

    return new PartParameter(
                             (partDescriptor != null && isNotBlank(partDescriptor.getFriendlyName()))
                                 ? partDescriptor.getFriendlyName() : apiParameterModel.getDisplayName(),
                             apiParameterModel.getExternalName(),
                             partTypeDefinition,
                             (partDescriptor != null && isNotBlank(partDescriptor.getDescription()))
                                 ? partDescriptor.getDescription() : apiParameterModel.getDescription(),
                             partDescriptor != null && partDescriptor.isFilePart());
  }
}
