/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.webapi.util;

import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.webapi.util.ParserUtils.splitCaps;

import javax.validation.constraints.NotNull;

public class XmlUtils {

  private XmlUtils() {}

  public static boolean isXmlSchema(String schema) {
    return isNotBlank(schema) && schema.trim().startsWith("<");
  }

  /**
   * @param artifact The artifact string to be sanitized.
   * @return Returns a sanitized string that is a valid Maven POM artifact.
   */
  public static String removeMavenArtifactUnwantedCharacters(String artifact) {
    if (artifact == null) {
      return null;
    }

    return artifact.replaceAll("[^a-zA-Z0-9_\\-]", EMPTY);
  }

  /**
   * @param group The group string to be sanitized.
   * @return Returns a sanitized string that is a valid Maven POM group.
   */
  public static String removeMavenGroupUnwantedCharacters(String group) {
    if (group == null) {
      return null;
    }

    return group.replaceAll("[^a-zA-Z0-9\\._\\-]", EMPTY);
  }

  /**
   * @param version The version string to be sanitized.
   * @return Returns a sanitized string that is a valid Maven POM version.
   */
  public static String removeMavenVersionUnwantedCharacters(String version) {
    if (version == null) {
      return null;
    }

    return version.replaceAll("[^a-zA-Z0-9\\._\\-]", EMPTY);
  }

  private static String splitCapsWithHypens(String name) {
    return splitCaps(name, "-");
  }

  private static String removeXmlUnwantedCharacters(String name) {
    if (name == null) {
      return null;
    }

    return name.replaceAll("[^a-zA-Z0-9_-]", EMPTY);
  }


  /**
   * @param name The string to be sanitized. Not Null.
   * @return Return a valid xml element name. It's format will be an underscore split lowercase string.
   */
  public static String getXmlName(@NotNull String name) {
    name = splitCapsWithHypens(name).toLowerCase();

    while (name.matches(".*[a-zA-Z0-9]+[\\s]+[a-zA-Z0-9]+.*")) {
      name = name.replaceAll("([a-zA-Z0-9])[\\s]+([a-zA-Z0-9])", "$1-$2");
    }

    name = removeXmlUnwantedCharacters(name);

    name = name.replaceAll("_+", "-");

    if (isEmpty(name)) {
      return EMPTY;
    }

    if (name.toLowerCase().startsWith("xml") || Character.isDigit(name.charAt(0))) {
      name = "_" + name;
    }

    if (name.startsWith("-")) {
      name = name.replaceFirst("-", EMPTY);
    }

    if (name.endsWith("-")) {
      name = name.substring(0, name.length() - 1);
    }

    return name;
  }
}
