/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel.parameter;

import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.util.HashCodeUtil;

import static org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils.getXmlName;

import java.util.List;

public class Parameter {

  /**
   * XML name used to identify this parameter internally.
   */
  private final String internalName;

  /**
   * Name used to identify this parameter externally.
   * i.e. the name of the query parameter the API is expecting.
   */
  private final String externalName;

  /**
   * User friendly name for this parameter
   */
  private final String displayName;

  private final String description;
  private final boolean required;
  private final String defaultValue;
  private final boolean isPassword;
  private final TypeDefinition typeDefinition;

  public Parameter(String displayName,
                   String externalName,
                   ParameterType parameterType,
                   TypeDefinition typeDefinition,
                   String description,
                   boolean required,
                   String defaultValue,
                   boolean password,
                   List<String> collisionInternalNames) {

    this.displayName = displayName;
    this.externalName = externalName;
    this.internalName = buildInternalName(externalName, collisionInternalNames, parameterType);

    this.typeDefinition = typeDefinition;
    this.isPassword = password;
    this.description = description;
    this.required = required;
    this.defaultValue = defaultValue;
  }

  private String buildInternalName(String externalName, List<String> collisionInternalNames, ParameterType parameterType) {
    String name = getXmlName(externalName);

    if (collisionInternalNames != null && collisionInternalNames.stream().anyMatch(x -> x.equalsIgnoreCase(name))) {
      return name + "-" + getXmlName(parameterType.getName());
    }

    return name;
  }

  public String getDisplayName() {
    return displayName;
  }

  public String getInternalName() {
    return internalName;
  }

  public String getExternalName() {
    return externalName;
  }

  public TypeDefinition getTypeDefinition() {
    return typeDefinition;
  }

  public String getDescription() {
    return description;
  }

  public boolean isRequired() {
    return required;
  }

  public String getDefaultValue() {
    return defaultValue;
  }

  public boolean isPassword() {
    return isPassword;
  }

  @Override
  public int hashCode() {
    return HashCodeUtil.generateHashCode(internalName, displayName, description, required, defaultValue, typeDefinition,
                                         isPassword);
  }

}
