/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating.sdk;

import static com.squareup.javapoet.MethodSpec.constructorBuilder;
import static com.squareup.javapoet.MethodSpec.methodBuilder;
import static com.squareup.javapoet.TypeSpec.anonymousClassBuilder;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PUBLIC;
import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.templating.JavaTemplateEntity;
import org.mule.runtime.extension.api.annotation.connectivity.oauth.OAuthParameter;
import org.mule.runtime.extension.api.annotation.param.NullSafe;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Summary;

import java.nio.file.Path;
import java.util.List;
import java.util.Map;

import static org.mule.connectivity.restconnect.internal.util.JavaUtils.getJavaLowerCamelNameFromXml;
import static org.mule.connectivity.restconnect.internal.util.JavaUtils.isReservedJavaWord;
import static org.mule.connectivity.restconnect.internal.util.JavaUtils.abbreviateText;
import static org.mule.connectivity.restconnect.internal.util.JavaUtils.removeJavaNameUnwantedCharacters;
import static org.mule.runtime.extension.api.util.NameUtils.pluralize;

public class SdkParameter extends JavaTemplateEntity {

  private final Parameter parameter;
  private final SdkTypeDefinition sdkTypeDefinition;
  private final String enumClassName;
  private final String enumClassPackage;

  public SdkParameter(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector, String parentJavaName,
                      Parameter parameter) {
    super(outputDir, connectorModel);
    this.parameter = parameter;
    this.sdkTypeDefinition = new SdkTypeDefinition(parameter.getTypeDefinition());
    this.enumClassName =
        sdkTypeDefinition.isEnum() ? buildEnumName(sdkConnector.getEnumValuesName(), parentJavaName, this) : null;
    this.enumClassPackage = connectorModel.getBasePackage() + ".api.metadata";
  }

  public String getExternalName() {
    return parameter.getExternalName();
  }

  public String getJavaName() {
    String javaName = getJavaLowerCamelNameFromXml(parameter.getInternalName());

    if (isArrayType()) {
      javaName = pluralize(javaName);
    }

    if (isReservedJavaWord(javaName)) {
      return "j" + javaName;
    } else {
      return javaName;
    }
  }

  public String getDisplayName() {
    return parameter.getDisplayName();
  }

  public String getDescription() {
    return parameter.getDescription();
  }

  public boolean isArrayType() {
    return sdkTypeDefinition.isArrayType();
  }

  private TypeName getTypeName() {
    return parameter.getTypeDefinition().isEnum() ? ClassName.get(enumClassPackage, enumClassName)
        : sdkTypeDefinition.getParameterTypeName();
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    if (sdkTypeDefinition.isEnum()) {
      generateEnumClass();
    }
  }

  private void generateEnumClass() throws TemplatingException {
    MethodSpec enumConstructor = constructorBuilder()
        .addParameter(String.class, "value")
        .addCode(CodeBlock.builder().addStatement("this.value = value").build())
        .build();

    MethodSpec valueGetterMethod = methodBuilder("getValue")
        .addModifiers(PUBLIC)
        .returns(String.class)
        .addCode(CodeBlock.builder().addStatement("return value").build())
        .build();

    TypeSpec.Builder enumClassBuilder = TypeSpec.enumBuilder(enumClassName)
        .addModifiers(PUBLIC)
        .addField(String.class, "value", PRIVATE)
        .addMethod(enumConstructor)
        .addMethod(valueGetterMethod);

    for (String enumConstant : parameter.getTypeDefinition().getEnumValues()) {
      enumClassBuilder.addEnumConstant(removeJavaNameUnwantedCharacters(enumConstant),
                                       anonymousClassBuilder("$S", enumConstant).build());
    }

    JavaFile.Builder javaFileBuilder = JavaFile
        .builder(enumClassPackage, enumClassBuilder.build());

    writeClassToFile(javaFileBuilder.build());
  }

  private static String buildEnumName(Map<List<String>, String> enumValuesName, String parentJavaName,
                                      SdkParameter sdkParameter) {
    List<String> enumValuesKey =
        enumValuesName.keySet().stream()
            .filter(x -> x.equals(sdkParameter.sdkTypeDefinition.getEnumValues()))
            .findFirst().orElse(null);

    //First, check if there is already an enum that defines the same values.
    //If it exists, use that name.
    if (enumValuesKey != null) {
      return enumValuesName.get(enumValuesKey);
    }

    //If not, try generating the enum value using the parameter name
    String parameterNameBasedEnumName = capitalize(sdkParameter.getJavaName()) + "Enum";
    if (!enumValuesName.containsValue(parameterNameBasedEnumName)) {
      enumValuesName.put(sdkParameter.sdkTypeDefinition.getEnumValues(), parameterNameBasedEnumName);
      return parameterNameBasedEnumName;
    }

    //If an enum using the parameter name already exists, use operation name + parameter name to generate the enum name
    String operationNameBasedEnumName = parentJavaName + parameterNameBasedEnumName;
    enumValuesName.put(sdkParameter.sdkTypeDefinition.getEnumValues(), operationNameBasedEnumName);
    return operationNameBasedEnumName;
  }

  public ParameterSpec.Builder generateParameterParameter() {
    ParameterSpec.Builder paramSpecBuilder = ParameterSpec
        .builder(getTypeName(), getJavaName());

    AnnotationSpec optionalAnnotation = getOptionalAnnotation();
    if (optionalAnnotation != null) {
      paramSpecBuilder.addAnnotation(optionalAnnotation);
    }

    AnnotationSpec nullSafeAnnotation = getNullSafeAnnotation();
    if (nullSafeAnnotation != null) {
      paramSpecBuilder.addAnnotation(nullSafeAnnotation);
    }

    paramSpecBuilder.addAnnotation(getDisplayNameAnnotation());

    AnnotationSpec summaryAnnotation = getSummaryAnnotation();
    if (summaryAnnotation != null) {
      paramSpecBuilder.addAnnotation(summaryAnnotation);
    }

    return paramSpecBuilder;
  }

  public FieldSpec.Builder generateParameterField() {
    AnnotationSpec parameterAnnotation = AnnotationSpec
        .builder(org.mule.runtime.extension.api.annotation.param.Parameter.class)
        .build();

    return generateParameterField(parameterAnnotation);
  }

  public FieldSpec.Builder generateOAuthParameterField() {
    AnnotationSpec parameterAnnotation = AnnotationSpec
        .builder(OAuthParameter.class)
        .addMember("requestAlias", "$S", parameter.getExternalName())
        .build();

    return generateParameterField(parameterAnnotation);
  }

  private FieldSpec.Builder generateParameterField(AnnotationSpec parameterAnnotation) {
    FieldSpec.Builder fieldSpecBuilder = FieldSpec
        .builder(getTypeName(), getJavaName())
        .addAnnotation(parameterAnnotation)
        .addAnnotation(getDisplayNameAnnotation());

    AnnotationSpec optionalAnnotation = getOptionalAnnotation();
    if (optionalAnnotation != null) {
      fieldSpecBuilder.addAnnotation(optionalAnnotation);
    }

    AnnotationSpec summaryAnnotation = getSummaryAnnotation();
    if (summaryAnnotation != null) {
      fieldSpecBuilder.addAnnotation(summaryAnnotation);
    }

    return fieldSpecBuilder;
  }

  private AnnotationSpec getNullSafeAnnotation() {
    if (!parameter.isRequired() && isArrayType()) {
      return AnnotationSpec.builder(NullSafe.class).build();
    }

    return null;
  }

  private AnnotationSpec getOptionalAnnotation() {
    if (!parameter.isRequired() || isNotBlank(parameter.getDefaultValue())) {
      AnnotationSpec.Builder optionalBuilder = AnnotationSpec.builder(Optional.class);

      if (isNotBlank(parameter.getDefaultValue())) {
        optionalBuilder.addMember("defaultValue", "$S", parameter.getDefaultValue());
      }

      return optionalBuilder.build();
    }

    return null;
  }

  private AnnotationSpec getDisplayNameAnnotation() {
    return AnnotationSpec
        .builder(DisplayName.class)
        .addMember(VALUE_MEMBER, "$S", parameter.getDisplayName())
        .build();
  }

  private AnnotationSpec getSummaryAnnotation() {
    if (isNotBlank(parameter.getDescription())) {
      return AnnotationSpec
          .builder(Summary.class)
          .addMember(VALUE_MEMBER, "$S", abbreviateText(parameter.getDescription()))
          .build();
    }
    return null;
  }

  public boolean isNullable() {
    return !sdkTypeDefinition.getParameterPrimitiveJavaType().equals(boolean.class);
  }

  public String getStringValueGetter() {
    return getStringValueGetter(getJavaName());
  }

  public String getStringValueGetter(String varName) {
    if (sdkTypeDefinition.isEnum()) {
      return varName + ".getValue()";
    } else if (sdkTypeDefinition.getJavaType().equals(String.class)) {
      return varName;
    } else {
      return "String.valueOf(" + varName + ")";
    }
  }
}
