package org.mule.connectivity.model.api;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.mule.connectivity.model.api.builder.RestConnectAPIModelBuilder;
import org.mule.connectivity.exception.*;
import org.mule.connectivity.model.operation.Operation;
import org.mule.connectivity.model.security.APISecurityScheme;
import org.raml.v2.api.RamlModelBuilder;
import org.raml.v2.api.RamlModelResult;
import org.raml.v2.api.model.v10.api.Api;
import org.raml.v2.api.model.v10.methods.Method;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang.StringUtils.join;
import static org.mule.connectivity.util.OperationMappingUtils.getMethods;
import static org.mule.connectivity.util.OperationMappingUtils.isIgnored;
import static org.mule.connectivity.util.OperationMappingUtils.disambiguateRepeatedOperations;
import static org.mule.connectivity.model.security.SecuritySchemeFactory.getSchemes;


public abstract class RestConnectAPIModel<T extends Operation> {

    public final static String DEFAULT_VERSION = "1.0.0-FD.1";
    public final static String VERSION_URI_PARAM = "{version}";
    public final static String VERSION_URI_PARAM_REGEX = "\\{version\\}";

    protected final static Logger logger = LogManager.getLogger(RestConnectAPIModel.class);

    protected final Api api;
    protected final List<T> operations;
    protected final List<APISecurityScheme> securitySchemes;

    protected final String apiName;
    protected String groupId;
    protected String artifactId;
    protected String version;


    public RestConnectAPIModel(RestConnectAPIModelBuilder<?, ?> builder) throws GenerationException {
        this.api = getAPIFromRamlFile(builder.getRaml());
        this.apiName = buildApiName(builder);

        this.groupId = buildGroupId(builder);
        this.artifactId = buildArtifactId(builder);
        this.version = buildVersion(builder);

        this.operations = buildOperationsModel();
        this.securitySchemes = buildSecurityModel(api);
    }

    private String buildApiName(RestConnectAPIModelBuilder<?, ?> builder) {
        if(!isBlank(builder.getApiName()))
            return builder.getApiName();

        return this.api.title().value();
    }

    private String buildVersion(RestConnectAPIModelBuilder<?, ?> builder) {
        if(!isBlank(builder.getVersion()))
            return builder.getVersion();

        return getDefaultVersion();
    }

    protected String getDefaultVersion() {
        return DEFAULT_VERSION;
    }

    private String buildGroupId(RestConnectAPIModelBuilder<?, ?> builder) {
        if(!isBlank(builder.getGroupId()))
            return builder.getGroupId();

        return getDefaultGroupId();
    }

    private String buildArtifactId(RestConnectAPIModelBuilder<?, ?> builder) {
        if(!isBlank(builder.getArtifactId()))
            return builder.getArtifactId();

        return getDefaultArtifactId();
    }

    private List<APISecurityScheme> buildSecurityModel(Api api) throws UnsupportedSecuritySchemeException {
        return getSchemes(api.securitySchemes());
    }

    /**
     * Iterate over the RAML files and build a org.mule.connectivity.model with all operations that will be used later
     *      by the template to applyTemplate code
     */
    public List<T> buildOperationsModel() throws GenerationException {
        final List<Method> methods = getMethods(api);
        List<T> operations = new ArrayList<>();

        for (Method method : methods) {
            if(isIgnored(method))
                logger.warn("Resource ignored: {} {}", method.method(), method.resource().resourcePath());
            else
                operations.add(createOperation(method));
        }

        disambiguateRepeatedOperations(operations);
        return operations;
    }

    public static Api getAPIFromRamlFile(File raml) {
        RamlModelResult ramlModelResult = new RamlModelBuilder().buildApi(raml);

        if(ramlModelResult.hasErrors()) {
            logger.error("Error detected in RAML: " + join(ramlModelResult.getValidationResults(), ", "));
            throw new InvalidRAMLSourceException("Invalid RAML.");
        }

        if(ramlModelResult.isVersion08()) {
            throw new InvalidRAMLSourceException("RAML 0.8 is not supported.");
        }

        if(ramlModelResult.getApiV10() == null)
            throw new InvalidRAMLSourceException("Invalid RAML: the provided source isn't an API definition but a fragment.");

        return ramlModelResult.getApiV10();
    }

    public List<APISecurityScheme> getSecuritySchemes() {
        return securitySchemes;
    }

    public String getBaseUri() {
        // Base cases. We don't deal with parameters, we leave this blank.
        if(api.baseUri() == null || !api.baseUriParameters().isEmpty())
            return "";

        // We replace parameters with the version if it is a template.
        if(api.baseUri().value().contains(VERSION_URI_PARAM) && api.version() != null && !api.version().value().isEmpty())
            return api.baseUri().value().replaceAll(VERSION_URI_PARAM_REGEX, api.version().value());

        // If it doesn't have the version then we return empty.
        else if(api.baseUri().value().contains(VERSION_URI_PARAM))
            return "";

        return api.baseUri().value();
    }

    public String getApiName() {
        return apiName;
    }

    public String getApiDescription() {
        return api.description() == null ? "" : api.description().value();
    }

    public List<T> getOperations() {
        return operations;
    }

    public String getGroupId() {
        return groupId;
    }

    public String getArtifactId() {
        return artifactId;
    }

    public String getVersion() {
        return version;
    }

    protected abstract T createOperation(Method method) throws UnsupportedMediaTypeException;

    protected abstract String getDefaultGroupId();

    protected abstract String getDefaultArtifactId();

}
