package org.mule.connectivity.model.api;

import org.mule.connectivity.model.MavenDependency;
import org.mule.connectivity.model.api.builder.SmartConnectorModelBuilder;
import org.mule.connectivity.exception.GenerationException;
import org.mule.connectivity.exception.UnsupportedMediaTypeException;
import org.mule.connectivity.model.operation.SmartConnectorOperation;
import org.raml.v2.api.model.v10.methods.Method;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.mule.connectivity.util.ParserUtils.getXmlName;


public class SmartConnectorModel extends RestConnectAPIModel<SmartConnectorOperation> {

    private final String projectDescription;
    private final URI baseUri;
    private final List<MavenDependency> dependencies;

    public SmartConnectorModel(SmartConnectorModelBuilder builder) throws GenerationException {
        super(builder);
        this.projectDescription = builder.getProjectDescription();
        this.baseUri = buildBaseUri();
        this.dependencies = builder.getDependencies();
    }

    private URI buildBaseUri() throws GenerationException {
        try {
            URI uri = new URI(getBaseUri());
            if(!uri.isAbsolute())
                return null;
            return uri;
        } catch (URISyntaxException e) {
            throw new GenerationException("Invalid base URI.", e);
        }
    }

    @Override
    protected SmartConnectorOperation createOperation(Method method) throws UnsupportedMediaTypeException {
        return new SmartConnectorOperation(method);
    }

    @Override
    protected String getDefaultGroupId() {
        return "org.mule.modules.extensions";
    }

    @Override
    protected String getDefaultArtifactId() {
        return "mule-module-" + getModulePrefix();
    }

    public String getModulePrefix() {
        return getXmlName(this.apiName);
    }

    public String getNamespace() {
        return "http://www.mulesoft.org/schema/mule/" + getModulePrefix();
    }

    public String getBasePath() {
        if(this.baseUri == null)
            return null;

        return this.baseUri.getPath();
    }

    public String getProtocol() {
        if(this.baseUri == null)
            return null;

        return this.baseUri.getScheme().toUpperCase();
    }

    public String getHost() {
        if(this.baseUri == null)
            return null;

        return this.baseUri.getHost();
    }

    public Integer getPort() {
        if(this.baseUri == null)
            return null;

        if(this.baseUri.getPort() != -1)
            return this.baseUri.getPort();

        switch (getProtocol()) {
            case "HTTP": return 80;
            case "HTTPS": return 443;
        }

        return null;
    }

    public List<MavenDependency> getDependencies() {
        return dependencies;
    }

    public String getProjectDescription() {
        if(!isBlank(projectDescription) && !isBlank(getApiDescription()))
            return format("%s - %s", projectDescription, getApiDescription());

        if(!isBlank(projectDescription))
            return projectDescription;

        if(!isBlank(getApiDescription()))
            return getApiDescription();

        return null;
    }

}
