package org.mule.connectivity.model.metadata;

import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.mule.connectivity.model.metadata.definition.SimpleTypeDefinition.PrimitiveType;
import org.mule.connectivity.model.metadata.model.JsonMetadataModel;
import org.mule.connectivity.model.metadata.model.MetadataModel;
import org.mule.connectivity.model.metadata.model.PrimitiveMetadataModel;
import org.mule.connectivity.model.metadata.model.XmlMetadataModel;
import org.mule.connectivity.predicate.HasBodyPredicate;
import org.mule.connectivity.predicate.IsSupportedTypePredicate;
import org.mule.connectivity.predicate.OkResponsePredicate;
import org.raml.v2.api.model.v10.bodies.Response;
import org.raml.v2.api.model.v10.datamodel.*;
import org.raml.v2.api.model.v10.methods.Method;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static org.mule.connectivity.model.metadata.PrimitiveMetadataModelFactory.createPrimitiveMetadataModel;


public class MetadataModelFactory {

    private static final Predicate<Response> okAndBodyPredicate = Predicates.and(new OkResponsePredicate(), new HasBodyPredicate());
    private static final Logger logger = LogManager.getLogger(MetadataModelFactory.class);

    private final Method method;

    public MetadataModelFactory(Method method) {
        this.method = method;
    }

    public MetadataModel constructInputMetadata(String modelName) {
        return getModel(modelName, method.body());
    }

    public MetadataModel constructOutputMetadata(String modelName) {
        List<TypeDeclaration> mediaTypes = new ArrayList<>();
        for (Response response : Iterables.filter(method.responses(), okAndBodyPredicate)) {
            mediaTypes.addAll(Collections2.filter(response.body(), new IsSupportedTypePredicate()));
        }

        return getModel(modelName, mediaTypes);
    }

    protected MetadataModel getModel(String className, List<TypeDeclaration> typeDeclarations) {
        Collection<TypeDeclaration> supportedTypeDeclarations = getTypeDeclarations(typeDeclarations);
        if (supportedTypeDeclarations == null)
            return null;

        TypeDeclaration typeDeclaration = supportedTypeDeclarations.iterator().next();

        if(typeIsDefinedWithRAML(typeDeclaration)) {
            return safelyGetMetadataModelFromRAMLType(className, typeDeclaration);
        }

        else if (typeIsDefinedWithXMLSchema(typeDeclaration)) {
            XMLTypeDeclaration xmlMediaType = (XMLTypeDeclaration) typeDeclaration;
            return new XmlMetadataModel(className, xmlMediaType.schemaContent(), typeDeclaration.name());
        }

        else if (typeIsDefinedWithJSONSchema(typeDeclaration)) {
            JSONTypeDeclaration jsonMediaType = (JSONTypeDeclaration) typeDeclaration;
            return new JsonMetadataModel(className, jsonMediaType.schemaContent(), typeDeclaration.name());
        }

        else if (typeIsPrimitive(typeDeclaration)) {
            return createPrimitiveMetadataModel(typeDeclaration.name(), typeDeclaration);
        }

        else if (typeDefinitionIsNotProvided(typeDeclaration)) {
            return new PrimitiveMetadataModel(typeDeclaration.name(), PrimitiveType.STRING);
        }

        // This shouldn't happen as it's already validated that the declaration is supported.
        else {
            logger.warn("Type {} cannot be parsed for operation {} {}.", typeDeclaration.type(), method.resource().resourcePath(), method.method());
        }

        return null;
    }

    private MetadataModel safelyGetMetadataModelFromRAMLType(String className, TypeDeclaration typeDeclaration) {
        // The XML SDK won't support RAML types until MULE-11501 is done.
        // There was a workaround implemented for translating those into JSON schemas implemented in RESTC-156 and RESTC-164.
        // There are known cases that are failing due to inconsistencies in the transformation between RAML types and
        // JSON schemas, so this is for preventing that the Smart Connector can be generated even though a single type
        // throws an exception.
        try {
            return new JsonMetadataModel(className, typeDeclaration.toJsonSchema(), typeDeclaration.name());
        }

        catch (RuntimeException exception) {
            logger.warn("Couldn't generate a type definition for " + className, exception);

            // In the future this should be changed to type "any" (RESTC-176).
            return new PrimitiveMetadataModel(typeDeclaration.name(), "string");
        }
    }

    private Collection<TypeDeclaration> getTypeDeclarations(List<TypeDeclaration> typeDeclarations) {
        if (typeDeclarations == null) {
            return null;
        }

        Collection<TypeDeclaration> supportedTypeDeclarations = Collections2.filter(typeDeclarations, new IsSupportedTypePredicate());

        if (supportedTypeDeclarations.isEmpty())
            return null;

        if (supportedTypeDeclarations.size() > 1) {
            logger.warn("Operation {} {} contains more than one media type. Found: {} media types. Will use the first one.",
                    method.resource().resourcePath(), method.method(), typeDeclarations.size());
        }

        return supportedTypeDeclarations;
    }

    public static boolean typeIsDefinedWithRAML(TypeDeclaration typeDeclaration) {
        return typeDeclaration instanceof ObjectTypeDeclaration || typeDeclaration instanceof ArrayTypeDeclaration || typeDeclaration instanceof UnionTypeDeclaration;
    }

    public static boolean typeIsDefinedWithXMLSchema(TypeDeclaration typeDeclaration) {
        return typeDeclaration instanceof XMLTypeDeclaration;
    }

    public static boolean typeIsDefinedWithJSONSchema(TypeDeclaration typeDeclaration) {
        return typeDeclaration instanceof JSONTypeDeclaration;
    }

    public static boolean typeIsPrimitive(TypeDeclaration typeDeclaration) {
        return typeDeclaration instanceof BooleanTypeDeclaration ||
                typeDeclaration instanceof DateTimeOnlyTypeDeclaration ||
                typeDeclaration instanceof DateTypeDeclaration ||
                typeDeclaration instanceof DateTimeTypeDeclaration ||
                typeDeclaration instanceof FileTypeDeclaration ||
                typeDeclaration instanceof IntegerTypeDeclaration ||
                typeDeclaration instanceof NumberTypeDeclaration ||
                typeDeclaration instanceof StringTypeDeclaration ||
                typeDeclaration instanceof TimeOnlyTypeDeclaration;
    }

    // This is for the case that the type is not defined (for instance, when there is an example but not a definition).
    public static boolean typeDefinitionIsNotProvided(TypeDeclaration typeDeclaration) {
        return typeDeclaration instanceof AnyTypeDeclaration;
    }

}
