package org.mule.connectivity.model.operation;

import com.google.common.collect.FluentIterable;
import com.google.common.collect.Iterables;
import org.mule.connectivity.exception.UnsupportedMediaTypeException;
import org.mule.connectivity.model.parameter.PropertyParameter;
import org.mule.connectivity.model.metadata.model.MetadataModel;
import org.raml.v2.api.model.v10.datamodel.TypeDeclaration;
import org.raml.v2.api.model.v10.methods.Method;
import org.raml.v2.api.model.v10.resources.Resource;

import java.util.*;

import static com.google.common.collect.Iterables.transform;
import static com.google.common.collect.Lists.newArrayList;
import static org.mule.connectivity.util.ParserUtils.getCanonicalOperationName;
import static org.mule.connectivity.util.ParserUtils.getMethodDescription;
import static org.mule.connectivity.util.ParserUtils.getMethodDisplayName;


public abstract class Operation {

    protected final String friendlyName;
    protected final String description;
    protected String canonicalName;
    protected final List<PropertyParameter> parameters;
    protected final List<PropertyParameter> uriParameters;
    protected final List<PropertyParameter> queryParameters;
    protected final List<PropertyParameter> headers;
    protected MetadataModel inputMetaData;
    protected MetadataModel outputMetaData;
    protected final String httpMethod;
    protected final String uri;


    public Operation(Method method) throws UnsupportedMediaTypeException {
        friendlyName = getMethodDisplayName(method);
        description = getMethodDescription(method);
        canonicalName = getCanonicalOperationName(method, this);
        httpMethod = method.method();
        uriParameters = buildUriParams(method);
        queryParameters = buildQueryParams(method);
        headers = buildHeaders(method);
        uri = method.resource().resourcePath();
        parameters = FluentIterable.from(Iterables.concat(queryParameters, uriParameters, headers)).toList();
    }

    private List<PropertyParameter> buildQueryParams(Method method) {
        return getParameterList(method.queryParameters());
    }

    private List<PropertyParameter> buildUriParams(Method method) {
        List<PropertyParameter> uriParameters = new ArrayList<>();

        // Due to RESTC-34, this is done recursively to the top.
        Resource resource = method.resource();
        while(resource != null) {
            uriParameters.addAll(0, getParameterList(resource.uriParameters()));
            resource = resource.parentResource();
        }

        return uriParameters;
    }

    private List<PropertyParameter> buildHeaders(Method method) {
        return getParameterList(method.headers());
    }

    private ArrayList<PropertyParameter> getParameterList(List<TypeDeclaration> typeDeclarationParameters) {
        return newArrayList(transform(typeDeclarationParameters, new PropertyParameter.ParameterTransformFunction()));
    }

    public boolean hasQueryParameters() {
        return !this.queryParameters.isEmpty();
    }

    public boolean hasUriParameters() {
        return !this.uriParameters.isEmpty();
    }

    public boolean hasHeaders() {
        return !this.headers.isEmpty();
    }

    public List<PropertyParameter> getQueryParameters() {
        return queryParameters;
    }

    public List<PropertyParameter> getUriParameters() {
        return uriParameters;
    }

    public List<PropertyParameter> getHeaders() {
        return headers;
    }

    public String getHttpMethod() {
        return httpMethod;
    }

    public String getUri() {
        return uri;
    }

    public Iterable<PropertyParameter> getParameters() {
        return parameters;
    }

    public String getFriendlyName() {
        return friendlyName;
    }

    public String getDescription() {
        return description;
    }

    public String getCanonicalName() {
        return canonicalName;
    }

    public void setCanonicalName(String canonicalName) {
        this.canonicalName = canonicalName;
    }

    public MetadataModel getInputMetaData() {
        return inputMetaData;
    }

    public MetadataModel getOutputMetaData() {
        return outputMetaData;
    }

}
