package org.mule.connectivity.model.parameter;

import com.google.common.base.Function;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.mule.connectivity.model.metadata.definition.JsonTypeDefinition;
import org.mule.connectivity.model.metadata.definition.SimpleTypeDefinition;
import org.mule.connectivity.model.metadata.definition.SimpleTypeDefinition.PrimitiveType;
import org.mule.connectivity.model.metadata.definition.TypeDefinition;
import org.mule.connectivity.util.ParserUtils;
import org.raml.v2.api.model.v10.datamodel.TypeDeclaration;

import javax.annotation.Nullable;

import static org.apache.commons.lang3.StringUtils.trimToNull;
import static org.mule.connectivity.model.metadata.MetadataModelFactory.typeIsDefinedWithRAML;
import static org.raml.v2.internal.utils.Inflector.lowercamelcase;
import static org.raml.v2.internal.utils.Inflector.uppercamelcase;


public class PropertyParameter {

    private static final Logger logger = LogManager.getLogger(PropertyParameter.class);

    public static final class ParameterTransformFunction implements Function<TypeDeclaration, PropertyParameter> {

        @Nullable
        @Override
        public PropertyParameter apply(@Nullable TypeDeclaration input) {
            return new PropertyParameter(input);
        }

    }

    private final TypeDefinition type;
    private final String name;
    private final boolean required;
    private final String description;

    public PropertyParameter(TypeDeclaration input) {
        this.type = buildType(input);
        this.name = input.name();
        this.required = input.required();
        this.description = input.description() == null ? null : trimToNull(input.description().value());
    }

    private TypeDefinition buildType(TypeDeclaration input) {
        // TODO (RESTC-141): this piece of code is similar to MetadataModelFactory::safelyGetMetadataModelFromRAMLType,
        // as there needs to be a check to see if the RAML type can be represented with a JSON schema.
        // Find a way to get the same logic in both places when returning either TypeDefinitions or a MetadataModel
        // instance. Considering merging those classes into a single one (after all, the Java RAML parser uses just
        // one type for both (TypeDeclaration).
        if(typeIsDefinedWithRAML(input))
            return safelyGetTypeDefinitionFromRAMLType(input);

        return new SimpleTypeDefinition(PrimitiveType.fromString(input.type()));
    }

    private TypeDefinition safelyGetTypeDefinitionFromRAMLType(TypeDeclaration input) {
        try {
            return new JsonTypeDefinition(input.name(), input.toJsonSchema());
        }

        catch (RuntimeException e) {
            logger.warn("Couldn't generate a type definition for " + input.name());
            return new SimpleTypeDefinition(PrimitiveType.fromString(input.type()));
        }
    }

    public String getName() {
        return name;
    }

    public String getJavaName() {
        return lowercamelcase(name);
    }

    public String getJavaClassName() {
        return uppercamelcase(name);
    }

    public String getXmlName() {
        return ParserUtils.getXmlName(name);
    }

    public TypeDefinition getType() {
        return type;
    }

    public String getDescription() {
        return description;
    }

    public boolean isRequired() {
        return required;
    }

}
