package org.mule.connectivity.templateEngine;

import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.velocity.VelocityContext;
import org.mule.connectivity.model.api.ConnectorModel;
import org.mule.connectivity.model.metadata.definition.CustomTypeDefinition;
import org.mule.connectivity.model.operation.ConnectorOperation;
import org.mule.connectivity.model.parameter.PropertyParameter;
import org.mule.connectivity.model.security.APISecurityScheme;
import org.mule.connectivity.templateEngine.builder.DevKitTemplateEngineBuilder;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.nio.file.Paths;

import static org.raml.v2.internal.utils.Inflector.uppercamelcase;


public class DevKitTemplateEngine extends TemplateEngine {

    private static final String CONNECTOR = "connector";
    private static final String OPERATION = "operation";
    private static final String BASE_URI = "baseUri";

    private static final String CONNECTOR_TEMPLATE_VM = "templates/devkit/ConnectorTemplate.vm";
    private static final String REST_CLIENT_CONFIG_PROVIDER_VM = "templates/devkit/RestClientConfigProvider.vm";
    private static final String FUNCTIONAL_TEST_CASE_VM = "templates/devkit/FunctionalTestCase.vm";
    private static final String ABSTRACT_TEST_CASE_VM = "templates/devkit/AbstractTestCases.vm";
    private static final String ABSTRACT_CONFIG_VM = "templates/devkit/AbstractConfig.vm";
    private static final String POM_VM = "templates/devkit/pom.vm";
    private static final String LICENSE_MD_VM = "templates/devkit/LICENSE.md";

    private static final Path MAIN_DIR = Paths.get("src/main/java/");
    private static final Path TEST_DIR = Paths.get("src/test/java/");

    private final ConnectorModel model;
    private final Path outputDir;
    private final boolean generateProjectFiles;

    private final Logger logger = LogManager.getLogger(DevKitTemplateEngine.class);


    public DevKitTemplateEngine(DevKitTemplateEngineBuilder builder) {
        this.model = builder.getModel();
        this.generateProjectFiles = builder.getProjectFilesGeneration();
        this.outputDir = builder.getOutputDir();
    }


    @Override
    public void applyTemplates() throws Exception {
        // Create POJOS
        this.createPOJOs();

        // Apply Templates
        this.createConnector();
    }

    private void createPOJOs() throws URISyntaxException {
        Path output = outputDir.resolve(MAIN_DIR);
        output.toFile().mkdirs();
        for (ConnectorOperation op : model.getOperations()) {
            try {
                if(op.getInputMetaData() != null && op.getInputMetaData().getMetadataDefinition() != null)
                    op.getInputMetaData().getMetadataDefinition().generatePojo(op.getMethodName() + "Request", output.toFile(), model.getBasePackage());
                if(op.getOutputMetaData() != null && op.getOutputMetaData().getMetadataDefinition() != null)
                    op.getOutputMetaData().getMetadataDefinition().generatePojo(op.getMethodName() + "Response", output.toFile(), model.getBasePackage());

                for(PropertyParameter parameter : op.getParameters()) {
                    if(parameter.getType().isCustomType())
                        ((CustomTypeDefinition) parameter.getType()).generatePojo(op.getMethodName() + parameter.getJavaClassName(), output.toFile(), model.getBasePackage());
                }
            } catch (IOException e) {
                logger.warn("Error while creating POJO file: %s", op.getMethodName());
            }
        }
    }

    private void createConnector() throws Exception {
        VelocityContext context = new VelocityContext();
        context.internalPut(CONNECTOR, model);

        if (StringUtils.isNotBlank(model.getBaseUri())) {
            context.internalPut(BASE_URI, model.getBaseUri());
        }

        Path outputBaseDir = outputDir;
        Path packagePath = Paths.get(model.getBasePackage().replace('.', File.separatorChar));
        Path mainDir = outputBaseDir.resolve(MAIN_DIR).resolve(packagePath);
        Path testDir = outputBaseDir.resolve(TEST_DIR).resolve(packagePath).resolve("automation/functional");

        for (APISecurityScheme securityScheme : model.getSecuritySchemes()) {
            applyTemplate(securityScheme.getTemplateLocation(), mainDir.resolve(securityScheme.getConfigurationName()), context);
        }

        applyTemplate(CONNECTOR_TEMPLATE_VM, mainDir.resolve(model.getClassName() + ".java"), context);
        applyTemplate(ABSTRACT_CONFIG_VM, mainDir.resolve("AbstractConfig.java"), context);
        applyTemplate(REST_CLIENT_CONFIG_PROVIDER_VM, mainDir.resolve("RestClientConfigProvider.java"), context);
        applyTemplate(LICENSE_MD_VM, outputBaseDir.resolve("LICENSE.md"), context);

        // Generate pom.xml only when project file generation is enabled.
        if(generateProjectFiles) {
            applyTemplate(POM_VM, outputBaseDir.resolve("pom.xml"), context);
        }

        // Functional tests
        applyTemplate(ABSTRACT_TEST_CASE_VM, testDir.resolve("AbstractTestCases.java"), context);

        for (ConnectorOperation op : model.getOperations()) {
            context.internalPut(OPERATION, op);
            applyTemplate(FUNCTIONAL_TEST_CASE_VM, testDir.resolve(op.getTestClassName() + ".java"), context);
        }
    }

}
