package org.mule.connectivity.templateEngine;

import org.apache.velocity.VelocityContext;
import org.apache.velocity.tools.ToolManager;
import org.mule.connectivity.exception.UnsupportedSecuritySchemeException;
import org.mule.connectivity.model.api.SmartConnectorModel;
import org.mule.connectivity.model.metadata.definition.CustomTypeDefinition;
import org.mule.connectivity.model.operation.SmartConnectorOperation;
import org.mule.connectivity.model.parameter.PropertyParameter;
import org.mule.connectivity.model.security.APISecurityScheme;
import org.mule.connectivity.model.security.BasicAuthScheme;
import org.mule.connectivity.model.security.OAuth2Scheme;
import org.mule.connectivity.templateEngine.builder.SmartConnectorTemplateEngineBuilder;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Properties;

import static java.lang.String.format;


public class SmartConnectorTemplateEngine extends TemplateEngine {

    private static final String SUPPORTS_BASIC_AUTH = "SUPPORTS_BASIC_AUTH";
    private static final String SUPPORTS_OAUTH2 = "SUPPORTS_OAUTH2";
    private static final String SUPPORTS_OAUTH2_CLIENT_CREDENTIALS = "SUPPORTS_OAUTH2_CLIENT_CREDENTIALS";
    private static final String EXTENSION = "extension";

    private static final String CATALOG_VM = "templates/smartConnector/catalog.vm";
    private static final String MODULE_VM = "templates/smartConnector/module.vm";
    private static final String POM_VM = "templates/smartConnector/pom.vm";

    private static final Path RESOURCES_DIR = Paths.get("src/main/resources");

    private static final String PROPERTIES_RESOURCE = "/project.properties";
    private static final String APPLICATION_PROPERTIES = "properties";

    private final SmartConnectorModel model;
    private final Path outputDir;


    public SmartConnectorTemplateEngine(SmartConnectorTemplateEngineBuilder builder) {
        this.model = builder.getModel();
        this.outputDir = builder.getOutputDir();
    }

    @Override
    public void applyTemplates() throws Exception {
        generateXmls();
        generateCatalog();
    }

    private void generateCatalog() throws IOException {
        Path outputResourcesDir = outputDir.resolve(RESOURCES_DIR);
        for (SmartConnectorOperation operation : this.model.getOperations()) {
            if(operation.getInputMetaData() != null && operation.getInputMetaData().getMetadataDefinition() != null) {
                operation.getInputMetaData().getMetadataDefinition().writeSchema(outputResourcesDir);
            }

            if(operation.getOutputMetaData() != null && operation.getOutputMetaData().getMetadataDefinition() != null) {
                operation.getOutputMetaData().getMetadataDefinition().writeSchema(outputResourcesDir);
            }

            for(PropertyParameter parameter : operation.getParameters()) {
                if(parameter.getType().isCustomType())
                    ((CustomTypeDefinition) parameter.getType()).writeSchema(outputResourcesDir);
            }
        }
    }

    private void generateXmls() throws Exception {
        ToolManager velocityToolManager = new ToolManager();
        velocityToolManager.configure("velocity-tools.xml");
        VelocityContext context = new VelocityContext(velocityToolManager.createContext());

        context.internalPut(EXTENSION, model);
        context.internalPut(APPLICATION_PROPERTIES, getApplicationProperties());

        // TODO: refactor!!
        for (APISecurityScheme securityScheme : model.getSecuritySchemes()) {
            if(securityScheme instanceof BasicAuthScheme)
                context.internalPut(SUPPORTS_BASIC_AUTH, true);
            if(securityScheme instanceof OAuth2Scheme) {
                context.internalPut(SUPPORTS_OAUTH2, true);
                if(((OAuth2Scheme) securityScheme).supportsAuthorizationGrant("client_credentials")) {
                    context.internalPut(SUPPORTS_OAUTH2_CLIENT_CREDENTIALS, true);
                }
            }
        }

        // Fails when none of the provided OAuth2 grant types are supported and there is no Basic Auth alternative.
        if(!(context.internalContainsKey(SUPPORTS_BASIC_AUTH))
                && context.internalContainsKey(SUPPORTS_OAUTH2)
                && !(context.internalContainsKey(SUPPORTS_OAUTH2_CLIENT_CREDENTIALS)) )
                    throw new UnsupportedSecuritySchemeException("Only OAuth 2.0 grant type supported is Client Credentials.");

        Path outputResourcesDir = outputDir.resolve(RESOURCES_DIR);

        applyTemplate(POM_VM, outputDir.resolve("pom.xml"), context);
        applyTemplate(MODULE_VM, outputResourcesDir.resolve(format("module-%s.xml", model.getModulePrefix())), context);
        applyTemplate(CATALOG_VM, outputResourcesDir.resolve(format("module-%s-catalog.xml", model.getModulePrefix())), context);
    }

    private Properties getApplicationProperties() throws IOException {
        try(InputStream input = this.getClass().getResourceAsStream(PROPERTIES_RESOURCE)) {
            Properties prop = new Properties();
            prop.load(input);
            return prop;
        }
    }

}
