package org.mule.connectivity.util;

import com.google.common.base.Function;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.TreeTraverser;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.mule.connectivity.model.operation.Operation;
import org.raml.v2.api.model.v10.api.Api;
import org.raml.v2.api.model.v10.declarations.AnnotationRef;
import org.raml.v2.api.model.v10.methods.Method;
import org.raml.v2.api.model.v10.resources.Resource;

import javax.annotation.Nullable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static java.lang.String.format;


public class OperationMappingUtils {

    protected final static Logger logger = LogManager.getLogger(OperationMappingUtils.class);

    private static final TreeTraverser<Resource> TRAVERSER = new TreeTraverser<Resource>() {

        @Override
        public Iterable<Resource> children(Resource root) {
            return root.resources();
        }

    };

    private static final String IGNORE_ANNOTATION = "ignored";

    /**
     * Traverses the RAML and extracts all actions that will eventually become operations
     * @param api
     */
    public static List<Method> getMethods(Api api) {
        List<Resource> resources = api.resources();

        return FluentIterable.from(resources).transformAndConcat(new Function<Resource, Iterable<Resource>>() {

            @Nullable
            @Override
            public Iterable<Resource> apply(@Nullable Resource input) {
                return TRAVERSER.preOrderTraversal(input);
            }

        }).transformAndConcat(new Function<Resource, Iterable<Method>>() {

            @Nullable
            @Override
            public Iterable<Method> apply(@Nullable Resource input) {
                return input.methods();
            }

        }).toList();
    }

    public static boolean isIgnored(Method method) {
        for (AnnotationRef annotationRef : method.annotations()) {
            if(IGNORE_ANNOTATION.equals(annotationRef.annotation().name()))
                return true;
        }

        return false;
    }

    /**
     * Renames the operations that have names that repeat appending a numbered suffix:
     *      - GET_STREET    ->  GET_STREET_1
     *      - GET_STREET    ->  GET_STREET_2
     *      - GET_STREET    ->  GET_STREET_3
     *      etc.
     *
     */
    public static void disambiguateRepeatedOperations(List<? extends Operation> operations) {
        // This set contains only the repeated names.
        Set<String> repeatedNames = new HashSet<>();

        // This map is used so that operations have an increasing number according to the position on the RAML.
        Map<String, Integer> replacedNames = new HashMap<>();

        for (Operation operation : operations) {
            String name = operation.getCanonicalName();

            if (replacedNames.containsKey(name)) {
                repeatedNames.add(name);
            }

            else {
                replacedNames.put(name, 0);
            }
        }

        // Appends a suffix to repeated operations.
        for (Operation operation : operations) {
            String operationName = operation.getCanonicalName();

            if (!repeatedNames.contains(operationName))
                continue;

            int occurrences = replacedNames.get(operationName) + 1;
            operation.setCanonicalName(format("%s_%d", operationName, occurrences));
            replacedNames.put(operationName, occurrences);
        }
    }

}
