package org.mule.connectivity.util;

import com.google.common.collect.FluentIterable;
import org.mule.connectivity.model.HTTPMethod;
import org.mule.connectivity.model.operation.Operation;
import org.mule.connectivity.predicate.OperationNamePredicate;
import org.raml.v2.api.model.v10.declarations.AnnotationRef;
import org.raml.v2.api.model.v10.methods.Method;
import org.raml.v2.api.model.v10.system.types.AnnotableStringType;

import java.util.ArrayList;
import java.util.List;

import static com.google.common.collect.Iterables.filter;
import static java.util.Collections.reverse;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.join;
import static org.apache.commons.lang3.StringUtils.trimToNull;
import static org.raml.v2.internal.utils.Inflector.*;


public class ParserUtils {

    // Overrides default construction to avoiding instantiation.
    private ParserUtils() {

    }

    public static String getMethodDescription(Method method) {
        String description = null;
        if(method.description() != null)
            description = method.description().value();

        if(method.resource() != null && method.resource().description() != null)
            description = method.resource().description().value();

        return trimToNull(description);
    }

    public static String getMethodDisplayName(Method method) {
        String displayName = null;
        // This checks for the case that the displayName attribute isn't provided, as the parsers
        // defaults this to the method name, per the RAML 1.0 spec.
        if(!method.displayName().value().equals(method.method()))
            displayName = method.displayName().value();

        return trimToNull(displayName);
    }


    public static String getCanonicalOperationName(Method method, Operation operation) {
        // This method produces an intermediate operation name consisting of uppercase words separated by underscores.

        // First tries to use the operationName annotation if it exists.
        String annotatedOperationName = getAnnotatedOperationName(method);
        if(annotatedOperationName != null)
            return getCanonicalNameFromFriendlyName(annotatedOperationName);

        // If the display name was provided, then we use that.
        if(operation.getFriendlyName() != null)
            return getCanonicalNameFromFriendlyName(operation.getFriendlyName());

        // Otherwise, we need to produce it ourselves.
        return buildCanonicalOperationName(method);
    }

    private static String buildCanonicalOperationName(Method method) {
        // Builds the operation name according to the method and path.
        //      GET     /towns                               ->  getTowns
        //      GET     /towns/{id}                          ->  getTownById
        //      POST    /towns                               ->  createTown
        //      PUT     /towns/{id}/streets                  ->  updateStreetById
        //      DELETE  /towns/{townId}/streets/{streetId}   ->  deleteStreetByTownIdStreetId
        // Note that:
        //      - Only the last resource is listed (deleteStreet...)
        //      - All URI params are listed, from left to right (...ByTownIdStreetId)
        HTTPMethod httpMethod = HTTPMethod.fromMethod(method);
        StringBuilder operationNameBuilder = new StringBuilder().append(httpMethod.getVerbName()).append("_");

        // This holds the last resource.
        String resourceName = null;
        // We use this variable to indicate if the resource is a collection or an element for pluralization at the end.
        Boolean isElement = null;
        // This keeps all parameters.
        List<String> parameters = new ArrayList<>();

        // Collects the names of the remaining parameters (if any) and saves the resource name (if not found yet).
        String remainingPath = method.resource().resourcePath();
        do {
            String pathSegment = remainingPath.substring(remainingPath.lastIndexOf("/") + 1);

            // Updates the isElement variable according to the last resource of the segment.
            boolean hasParameter = hasUriParameter(pathSegment);
            if(isElement == null)
                isElement = hasParameter;

            // Appends the parameter to the list.
            if(hasParameter)
                parameters.add(getParameterName(pathSegment));

            // Note that if it is a parameter then it can't be a resource. This updates the resource.
            else if(resourceName == null)
                resourceName = pathSegment;

            remainingPath = remainingPath.substring(0, remainingPath.lastIndexOf("/"));
        } while (isNotBlank(remainingPath) && !remainingPath.equals("/"));

        // This handles the case of an endpoint defined in "/".
        if(isNotBlank(resourceName)) {
            // Converts the resource to singular / plural.
            if(isElement || httpMethod.isAlwaysSingular())
                resourceName = singularize(resourceName);
            else
                resourceName = pluralize(resourceName);

            operationNameBuilder.append(upperunderscorecase(resourceName));
        }

        // Appends all the parameters (in reverse order, as it should go from left to right).
        if(!parameters.isEmpty()) {
            reverse(parameters);
            operationNameBuilder.append("_BY_").append(join(parameters, "_"));
        }

        return operationNameBuilder.toString();
    }

    // Removes unwanted characters possibly present.
    public static String removeUnwantedCharacters(String name) {
        if(name == null)
            return null;

        return name.replaceAll("[^a-zA-Z0-9_-]", "");
    }

    public static String removePackageUnwantedCharacters(String name) {
        if(name == null)
            return null;

        return name.replaceAll("[^a-zA-Z0-9]", "").toLowerCase();
    }

    public static String getXmlName(String name) {
        return lowerhyphencase(getXmlSanitizedName(name));
    }

    public static String getXmlSanitizedName(String name) {
        name = removeUnwantedCharacters(name);

        if(name.toLowerCase().startsWith("xml") || name.startsWith("-"))
            name = "-" + name;

        return name;
    }

    public static String getJavaName(String operationName) {
        return lowercamelcase(removeUnwantedCharacters(operationName));
    }

    private static String getCanonicalNameFromFriendlyName(String friendlyName) {
        // Removes all sentence stuff.
        friendlyName = friendlyName.replace(" ", "_");
        return removeUnwantedCharacters(friendlyName.toUpperCase());
    }

    private static String getParameterName(String uriValue) {
        String cleanParameterName = uriValue.replace("{", "").replace("}", "");
        return upperunderscorecase(cleanParameterName);
    }

    private static boolean hasUriParameter(String uriValue) {
        return uriValue.matches("\\{.*\\}");
    }

    private static String getAnnotatedOperationName(Method method) {
        AnnotationRef opNameAnnotation = FluentIterable.from(filter(method.annotations(), new OperationNamePredicate())).first().orNull();
        if(opNameAnnotation == null)
            return null;

        return (String) opNameAnnotation.structuredValue().value();
    }

    public static String getValueFromAnnotableString(AnnotableStringType annotableString) {
        return annotableString == null ? null : annotableString.value();
    }

    public static String removeSpaces(String string) {
        return string == null ? null : string.replace(" ", "");
    }

    public static String removeHyphens(String string) {
        return string == null ? null : string.replace("-", "");
    }

    public static String removeSpacesAndHyphens(String string) {
        return removeHyphens(removeSpaces(string));
    }

}
