package org.mule.connectivity.restconnect.internal.templateEngine.decorator.operation;

import static java.lang.Integer.parseInt;
import static java.lang.System.getProperty;
import static java.lang.Thread.interrupted;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinitionBuilder;
import org.mule.connectivity.restconnect.internal.model.typesource.JsonTypeSource;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.type.TypeDefinitionDecorator;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.internal.utils.MetadataTypeWriter;
import org.mule.metadata.json.api.JsonTypeLoader;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public abstract class OperationDecorator {

    private final static Logger logger = LogManager.getLogger(OperationDecorator.class);

    protected final Operation operation;
    protected Boolean inputMetadataTooLong = null;
    protected Boolean outputMetadataTooLong = null;

    public OperationDecorator(Operation operation) {
        this.operation = operation;
    }

    public TypeDefinition getInputMetadata() {
        TypeDefinition inputMetadata = this.operation.getInputMetadata();

        if(inputMetadataTooLong == null){
            inputMetadataTooLong = isMetadataTooLong(inputMetadata);
        }

        if(inputMetadataTooLong){
            return TypeDefinitionBuilder.buildStringTypeWithMediaType(inputMetadata.getMediaType(), inputMetadata.isRequired());
        }
        else{
            return inputMetadata;
        }
    }

    public TypeDefinition getOutputMetadata() {
        TypeDefinition outputMetadata = this.operation.getOutputMetadata();

        if(outputMetadataTooLong == null){
            outputMetadataTooLong = isMetadataTooLong(outputMetadata);
        }

        if(outputMetadataTooLong){
            return TypeDefinitionBuilder.buildStringTypeWithMediaType(outputMetadata.getMediaType(), outputMetadata.isRequired());
        }
        else{
            return outputMetadata;
        }
    }

    private boolean isMetadataTooLong(TypeDefinition metadata){
        boolean validMetadata = true;

        if(metadata != null &&
            metadata.getSource() != null &&
            metadata.getSource() instanceof JsonTypeSource &&
            StringUtils.isNotBlank(metadata.getSource().getValue())){

            Callable<Boolean> isMetadataLoaded = () -> {
                JsonTypeLoader jsonTypeLoader = new JsonTypeLoader(metadata.getSource().getValue());
                Optional<MetadataType> metadataType1 = jsonTypeLoader.load(null);
                Optional<MetadataType> metadataType2 = jsonTypeLoader.load(null);

                if(metadataType1.isPresent()){
                    if(metadataType1.equals(metadataType2)){
                        return true;
                    }
                }

                if (interrupted()) {
                    return false;
                }

                return true;
            };

            ExecutorService executorService = Executors.newCachedThreadPool();
            Future<Boolean> isMetadataLoadedTask = executorService.submit(isMetadataLoaded);

            try {
                isMetadataLoadedTask.get(getMetadataTimeoutSeconds(), TimeUnit.MILLISECONDS);
                logger.debug("Metadata loaded. Length: " + metadata.getSource().getValue().length());
            } catch (InterruptedException | CancellationException | TimeoutException e) {
                logger.warn("Error: Metadata check timed out. Length: " + metadata.getSource().getValue().length());
                validMetadata = false;
            } catch (ExecutionException e) {
                logger.warn("Error: Invalid metadata.");
                validMetadata = false;
            } finally {
                executorService.shutdownNow();
            }
        }

        return !validMetadata;
    }

    private int getMetadataTimeoutSeconds() {
        int metadataTimeoutSeconds;
        String metadataTimeoutProperty = getProperty("metadataTimeout", "60");

        try{
            metadataTimeoutSeconds = parseInt(metadataTimeoutProperty);
        }
        catch (NumberFormatException e){
            metadataTimeoutSeconds = 60;
        }

        return Math.max(metadataTimeoutSeconds, 1);
    }

    public List<Parameter> getQueryParameters() {
        return this.operation.getQueryParameters();
    }

    public List<Parameter> getUriParameters() {
        return this.operation.getUriParameters();
    }

    public List<Parameter> getPartParameters() {
        return this.operation.getPartParameters();
    }

    public List<Parameter> getHeaders() {
        return this.operation.getHeaders();
    }

    public List<Parameter> getParameters() {
        return this.operation.getParameters();
    }

    public boolean hasQueryParameters() {
        return !this.getQueryParameters().isEmpty();
    }

    public boolean hasUriParameters() {
        return !this.getUriParameters().isEmpty();
    }

    public boolean hasHeaders() {
        return !this.getHeaders().isEmpty();
    }

    public String getCanonicalName() {
        return operation.getCanonicalName();
    }

    public String getFriendlyName() {
        return operation.getFriendlyName();
    }

    public String getAnnotatedDisplayName() {
        return operation.getAnnotatedDisplayName();
    }

    public String getHttpMethod() {
        return operation.getHttpMethod();
    }

    public String getUri() {
        return operation.getUri();
    }
}
