package org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.mule.connectivity.restconnect.api.SpecFormat;
import org.mule.connectivity.restconnect.exception.InvalidSourceException;
import org.mule.connectivity.restconnect.internal.model.api.RestConnectModelBuilder;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.uri.BaseUri;
import org.mule.connectivity.restconnect.internal.model.uri.BaseUriBuilder;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.ModelGenerator;
import org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.resourceLoader.RamlParserExchangeDependencyResourceLoader;
import org.raml.v2.api.RamlModelBuilder;
import org.raml.v2.api.RamlModelResult;
import org.raml.v2.api.loader.CompositeResourceLoader;
import org.raml.v2.api.loader.DefaultResourceLoader;
import org.raml.v2.api.model.v10.api.Api;
import org.raml.v2.api.model.v10.methods.Method;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import static org.apache.commons.io.Charsets.UTF_8;
import static org.apache.commons.lang3.StringUtils.join;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlOperationMappingUtils.getMethods;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils.isIgnored;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.OperationMappingUtils.disambiguateRepeatedOperations;

public class RamlParserModelGenerator extends ModelGenerator {
    private final static Logger logger = LogManager.getLogger(RamlParserModelGenerator.class);

    public RamlParserModelGenerator(SpecFormat specFormat) {
        super(specFormat);

        if(!specFormat.equals(SpecFormat.RAML)){
            throw new IllegalArgumentException("RamlParser does not support " + specFormat.getName() + " format");
        }
    }

    public RestConnectModelBuilder generateAPIModel(File raml, String rootDir) throws Exception {
        Api api = getAPIFromRamlFile(raml, rootDir);

        Path rootDirPath = new File(rootDir).toPath().toAbsolutePath();

        JsonSchemaPool jsonSchemaPool = new JsonSchemaPool();

        return RestConnectModelBuilder.createModel()
                .withRootDir(rootDirPath)
                .withApiDescription(api.description() != null ? api.description().value() : "")
                .withApiName(api.title() != null ? api.title().value() : "")
                .withBaseUri(buildBaseUri(api))
                .withOperations(buildOperationsModel(api, jsonSchemaPool));
    }

    private static BaseUri buildBaseUri(Api api) {
        String baseUriString = api.baseUri() != null ? api.baseUri().value() : "";
        String versionString = api.version() != null ? api.version().value() : "";

        return BaseUriBuilder.buildBaseUri(baseUriString, versionString);
    }

    private static List<Operation> buildOperationsModel(Api api, JsonSchemaPool jsonSchemaPool) throws Exception {
        final List<Method> methods = getMethods(api);
        List<Operation> operations = new ArrayList<>();

        for (Method method : methods) {
            if(isIgnored(method))
            {
                logger.warn("Resource ignored: {} {}", method.method(), method.resource().resourcePath());
            }
            else
            {
                operations.add(RamlParserOperationGenerator.generateOperation(api, method, jsonSchemaPool));
            }
        }

        disambiguateRepeatedOperations(operations);
        return operations;
    }

    public static Api getAPIFromRamlFile(File raml, String rootDir) throws IOException {
        if(isRaml08(raml.toPath().toAbsolutePath())){
            throw new InvalidSourceException("RAML 0.8 is not supported.");
        }

        final CompositeResourceLoader custom = new CompositeResourceLoader(new DefaultResourceLoader(), new RamlParserExchangeDependencyResourceLoader(rootDir));

        final RamlModelBuilder ramlModelBuilder = new RamlModelBuilder(custom);

        RamlModelResult ramlModelResult;
        try{
            ramlModelResult = ramlModelBuilder.buildApi(raml);
        }
        catch (Exception e){
            throw new InvalidSourceException("Invalid RAML: Error in RAML Parser: " + e.getMessage() + ".", e);
        }

        if(ramlModelResult.hasErrors()) {
            String ramlErrors = join(ramlModelResult.getValidationResults(), ", ");
            throw new InvalidSourceException("Invalid RAML: " + ramlErrors + ".");
        }

        if(ramlModelResult.isVersion08()) {
            throw new InvalidSourceException("RAML 0.8 is not supported.");
        }

        if(ramlModelResult.getApiV10() == null)
            throw new InvalidSourceException("Invalid RAML: the provided source isn't an API definition but a fragment.");

        return ramlModelResult.getApiV10();
    }

    private static boolean isRaml08(Path specPath) throws IOException {
        String content = new String(java.nio.file.Files.readAllBytes(specPath), UTF_8);;

        return content.startsWith("#%RAML 0.8");
    }
}
