package org.mule.connectivity.restconnect.internal.templateEngine.decorator.security.smartconnector;

import org.apache.commons.lang3.StringUtils;
import org.mule.connectivity.restconnect.internal.model.MavenDependency;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.model.security.APISecurityScheme;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.operation.SmartConnectorOperationDecorator;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.type.SmartConnectorTypeDefinitionDecorator;

import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static org.mule.connectivity.restconnect.internal.templateEngine.decorator.security.smartconnector.SmartConnectorSecuritySchemeDecoratorHelper.buildPrefixedParameters;

public abstract class SmartConnectorSecuritySchemeDecorator {

    protected final List<SmartConnectorXMLConfig> xmlConfigs;
    protected final List<SmartConnectorXMLSchema> xmlSchemas;
    protected final List<SmartConnectorXMLSchemaLocation> xmlSchemaLocations;
    protected final String name;
    protected final List<SmartConnectorTypeDefinitionDecorator> configProperties;
    protected final List<Parameter> headers;
    protected final List<Parameter> queryParameters;
    protected final List<SmartConnectorPomDependency> pomDependencies;
    protected final APISecurityScheme apiSecurityScheme;
    protected String configNameSuffix = "";

    public SmartConnectorSecuritySchemeDecorator(APISecurityScheme apiSecurityScheme, HashMap<APISecurityScheme, String> securitySchemesNames) {
        this.xmlConfigs = buildXmlConfigs();
        this.xmlSchemas = buildXmlSchemas();
        this.xmlSchemaLocations = buildXmlSchemaLocations();
        this.name = buildName(apiSecurityScheme, securitySchemesNames);
        this.configProperties = buildConfigParameters(apiSecurityScheme, name);
        this.headers = buildHeaders(apiSecurityScheme, name);
        this.queryParameters = buildQueryParameters(apiSecurityScheme, name);
        this.pomDependencies = buildPomDependencies();
        this.apiSecurityScheme = apiSecurityScheme;
    }

    private static List<Parameter> buildQueryParameters(APISecurityScheme securityScheme, String prefix) {
        return buildPrefixedParameters(securityScheme.getQueryParameters(), prefix, ParameterType.QUERY);
    }

    private static List<Parameter> buildHeaders(APISecurityScheme securityScheme, String prefix) {
        return buildPrefixedParameters(securityScheme.getHeaders(), prefix, ParameterType.HEADER);
    }

    private static List<SmartConnectorTypeDefinitionDecorator> buildConfigParameters(APISecurityScheme securityScheme, String prefix) {
        return SmartConnectorOperationDecorator.buildDecoratedParameters(securityScheme.getConfigParameters(), prefix, new HashMap<>());
    }

    private String buildName(APISecurityScheme securityScheme, HashMap<APISecurityScheme, String> securitySchemesNames) {
        if (securitySchemesNames.isEmpty()){
            securitySchemesNames.put(securityScheme, "");
            return "";
        }
        Optional<APISecurityScheme> existingScheme = securitySchemesNames.keySet().stream().filter(x -> x.equals(securityScheme)).findFirst();
        if(existingScheme.isPresent()){
            //If this scheme exists in the static list, we use the same name.
            return securitySchemesNames.get(existingScheme.get());
        }
        else {
            //If this scheme does not exist, we create a new one.
            String prefix = getPrefixForSchemeType(securityScheme.getSchemeName());
            String nameCandidate = prefix;

            Integer i = 2;
            while (securitySchemesNames.containsValue(nameCandidate)){
                nameCandidate = prefix + i.toString();
                this.configNameSuffix = i.toString();
                i ++;
            }

            //Add the created name in the static list
            securitySchemesNames.put(securityScheme, nameCandidate);
            return nameCandidate;
        }
    }

    private static String getPrefixForSchemeType(String schemeType){
        if(schemeType.equals(APISecurityScheme.BASIC)){
            return "basic-auth";
        }
        if(schemeType.equals(APISecurityScheme.OAUTH2)){
            return "oauth2";
        }
        if(schemeType.equals(APISecurityScheme.PASS_THROUGH)){
            return "pass-through";
        }
        if(schemeType.equals(APISecurityScheme.DIGEST_AUTHENTICATION)){
            return "digest";
        }
        if(schemeType.equals(APISecurityScheme.CUSTOM_AUTHENTICATION)){
            return "custom-auth";
        }
        if(schemeType.equals(APISecurityScheme.UNSECURED)){
            return "unsecured";
        }
        return "security-scheme";
    }

    protected abstract List<SmartConnectorXMLSchemaLocation> buildXmlSchemaLocations();

    protected abstract List<SmartConnectorXMLSchema> buildXmlSchemas();

    protected abstract List<SmartConnectorXMLConfig> buildXmlConfigs();

    protected abstract List<SmartConnectorPomDependency> buildPomDependencies();

    public String getXmlConfigsString(){
        return SmartConnectorSecuritySchemeDecoratorHelper.getXmlConfigsString(xmlConfigs);
    }

    public String getXmlSchemasString(){
        return SmartConnectorSecuritySchemeDecoratorHelper.getXmlSchemasString(xmlSchemas);
    }

    public String getXmlSchemaLocationsString(){
        return SmartConnectorSecuritySchemeDecoratorHelper.getXmlSchemaLocationsString(xmlSchemaLocations);
    }

    public List<MavenDependency> getPomDependencies(){
        return pomDependencies.stream().map(x -> x.getMavenDependency()).collect(Collectors.toList());
    }

    public abstract String getHttpAuthorizationConfigString();

    public String getName(){
        return name;
    }

    public List<SmartConnectorTypeDefinitionDecorator> getConfigProperties(){
        return configProperties;
    }

    public List<Parameter> getHeaders(){
        return headers;
    }

    public List<Parameter> getQueryParameters(){
        return queryParameters;
    }

    protected SmartConnectorTypeDefinitionDecorator getPrefixedProperty(String propertyName){
        Optional<SmartConnectorTypeDefinitionDecorator> parameter = getConfigProperties().stream().filter(x -> x.getExternalName().equalsIgnoreCase((name.isEmpty() ? "" : name + "-" ) + propertyName)).findFirst();
        return parameter.orElse(null);
    }

    protected String getConfigLineForProperty(String propertyName, String configName){
        SmartConnectorTypeDefinitionDecorator property = getPrefixedProperty(propertyName);
        return configName + "=\"#[vars." + property.getPropertyName() + "]\"" + System.lineSeparator();
    }

    public String getXmlHttpConfigString(String modulePrefix, boolean setDefaultConnection, boolean generateProxyConfig){
        StringBuilder builder = new StringBuilder();

        builder.append("<http:request-config").append(System.lineSeparator());
        builder.append("name=\"").append(modulePrefix).append("-httpreq-config");
        if (!(name.isEmpty())) {
            builder.append("-").append(name);
        }
        builder.append("\"");
        if(setDefaultConnection){
            builder.append(" xmlns:connection=\"true\"");
        }
        builder.append(System.lineSeparator());
        builder.append("basePath=\"#[vars.property_basePath]\">").append(System.lineSeparator());
        builder.append("<http:request-connection").append(System.lineSeparator());
        builder.append("host=\"#[vars.property_host]\"").append(System.lineSeparator());
        builder.append("protocol=\"#[vars.property_protocol]\"").append(System.lineSeparator());
        builder.append("port=\"#[vars.property_port]\"").append(System.lineSeparator());
        if(generateProxyConfig){
            builder.append("proxyConfig=\"#[vars.property_proxyConfig]\"").append(System.lineSeparator());
        }
        builder.append(">");
        builder.append(getHttpAuthorizationConfigString()).append(System.lineSeparator());
        builder.append("</http:request-connection>").append(System.lineSeparator());
        builder.append("</http:request-config>").append(System.lineSeparator());

        return builder.toString();
    }

    public boolean equals(SmartConnectorSecuritySchemeDecorator other) {
        return other.getName().equals(name);
    }

    public String getFriendlyName(){
        return apiSecurityScheme.getSchemeName() + (StringUtils.isNotBlank(configNameSuffix) ? (" " + configNameSuffix) : "");
    }

    public boolean hasHeaders(){
        return apiSecurityScheme.hasHeaders();
    }

    public boolean hasQueryParameters(){
        return apiSecurityScheme.hasQueryParameters();
    }
}
