package org.mule.connectivity.restconnect.internal.modelGeneration.amf;

import amf.ProfileNames;
import amf.client.AMF;
import amf.client.environment.DefaultEnvironment;
import amf.client.environment.Environment;
import amf.client.model.document.Document;
import amf.client.model.domain.EndPoint;
import amf.client.model.domain.Server;
import amf.client.model.domain.WebApi;
import amf.client.parse.Oas20Parser;
import amf.client.parse.Oas20YamlParser;
import amf.client.parse.Parser;
import amf.client.parse.Raml10Parser;
import amf.client.validate.ValidationReport;
import amf.core.exception.UnsupportedVendorException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.mule.connectivity.restconnect.api.SpecFormat;
import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.exception.InvalidSourceException;
import org.mule.connectivity.restconnect.internal.model.api.RestConnectModelBuilder;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.uri.BaseUri;
import org.mule.connectivity.restconnect.internal.model.uri.BaseUriBuilder;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.ModelGenerator;
import org.mule.connectivity.restconnect.internal.modelGeneration.amf.resourceLoader.AMFExchangeDependencyResourceLoader;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.isIgnored;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.OperationMappingUtils.disambiguateRepeatedOperations;

public class AMFModelGenerator extends ModelGenerator {
    private final static Logger logger = LogManager.getLogger(AMFModelGenerator.class);
    private static boolean amfInitialized = false;

    public AMFModelGenerator(SpecFormat specFormat) {
        super(specFormat);
    }

    public RestConnectModelBuilder generateAPIModel(File raml, String rootDir) throws GenerationException, ExecutionException, InterruptedException {
        initializeAMF();

        Parser parser = getParser(specFormat, rootDir);
        WebApi webApi = getWebApi(parser, raml.toPath());

        Path rootDirPath = new File(rootDir).toPath().toAbsolutePath();
        JsonSchemaPool jsonSchemaPool = new JsonSchemaPool();

        return RestConnectModelBuilder.createModel()
                .withRootDir(rootDirPath)
                .withApiDescription(webApi.description().nonEmpty() ? webApi.description().value() : "")
                .withApiName(webApi.name().value())
                .withBaseUri(buildBaseUri(webApi))
                .withOperations(buildOperationsModel(webApi, jsonSchemaPool));
    }

    private static void initializeAMF() throws ExecutionException, InterruptedException {
        if(!amfInitialized){
            AMF.init().get();
            amfInitialized = true;
        }
    }

    private static List<Operation> buildOperationsModel(WebApi webApi, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Operation> operations = new ArrayList<>();

        for (EndPoint endPoint : webApi.endPoints()) {
            for(amf.client.model.domain.Operation operation : endPoint.operations()){
                if(isIgnored(endPoint, operation))
                {
                    logger.warn("Resource ignored: {} {}", operation.method(), endPoint.path());
                }
                else
                {
                    operations.add(AMFOperationGenerator.generateOperation(webApi, endPoint, operation, jsonSchemaPool));
                }
            }
        }

        disambiguateRepeatedOperations(operations);
        return operations;
    }

    private static BaseUri buildBaseUri(WebApi api) {
        String url = "";

        for(Server server : api.servers()){
            if(server.url().nonEmpty()){
                url = server.url().value();
                break;
            }
        }

        return BaseUriBuilder.buildBaseUri(url, api.version().value());
    }

    private static Parser getParser(SpecFormat format, String rootDir) {
        final Environment env = DefaultEnvironment.apply().add(new AMFExchangeDependencyResourceLoader(rootDir));

        if(format.equals(SpecFormat.RAML)){
            return new Raml10Parser(env);
        }
        else if(format.equals(SpecFormat.JSON_OAS)){
            return new Oas20Parser(env);
        }
        else if(format.equals(SpecFormat.YAML_OAS)){
            return new Oas20YamlParser(env);
        }

        throw new IllegalArgumentException("Spec format " + format.getName() + "not supported");
    }

    public WebApi getWebApi(Parser parser, Path path){
        if(isRaml08(path.toAbsolutePath())){
            throw new InvalidSourceException("RAML 0.8 is not supported.");
        }

        Document ramlDocument;
        try{
            ramlDocument = parseFile(parser, "file://" + path.toAbsolutePath().toString());

            validate(ramlDocument, specFormat);

            ramlDocument = resolve(ramlDocument, specFormat);
        }
        catch (Exception e){
            throw new InvalidSourceException("Invalid Spec: Error in AMF: " + e.getMessage() + ".", e);
        }

        return getWebApi(ramlDocument);
    }

    private static Document resolve(Document ramlDocument, SpecFormat specFormat) {
        if(specFormat.equals(SpecFormat.RAML)){
            return (Document)AMF.resolveRaml10(ramlDocument);
        }
        else if(specFormat.equals(SpecFormat.JSON_OAS) || specFormat.equals(SpecFormat.YAML_OAS)){
            return  (Document)AMF.resolveOas20(ramlDocument);
        }
        throw new IllegalArgumentException("Unrecognised spec format");
    }

    private static void validate(Document ramlDocument, SpecFormat specFormat) throws ExecutionException, InterruptedException {
        ValidationReport report;

        if(specFormat.equals(SpecFormat.RAML)){
            report = AMF.validate(ramlDocument, ProfileNames.RAML(), ProfileNames.AMF().messageStyle()).get();
            if(!report.conforms()){
                throw new InvalidSourceException(report.toString());
            }
        }

        else if(specFormat.equals(SpecFormat.JSON_OAS) || specFormat.equals(SpecFormat.YAML_OAS)){
            report = AMF.validate(ramlDocument, ProfileNames.OAS(), ProfileNames.AMF().messageStyle()).get();
            if(!report.conforms()){
                throw new InvalidSourceException(report.toString());
            }
        }

        else{
            throw new IllegalArgumentException("Unrecognised spec format");

        }
    }

    public static WebApi getWebApi(Document document) {
        return (WebApi) document.encodes();
    }

    public static Document parseFile(Parser parser, String url) {
        return handleFuture(parser.parseFileAsync(url));
    }

    private static <T, U> U handleFuture(CompletableFuture<T> f) {
        try {
            return (U) f.get();
        } catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException("An error happend while parsing the api. Message: " + e.getMessage(), e);
        }
    }

    private boolean isRaml08(Path specPath){
        String content = "";

        try {
            content = new String(java.nio.file.Files.readAllBytes(specPath), "UTF-8");
        } catch (IOException e) {
            e.printStackTrace();
        }

        return content.startsWith("#%RAML 0.8");
    }

}
