package org.mule.connectivity.restconnect.internal.modelGeneration.amf;

import amf.client.model.domain.*;
import amf.plugins.xml.transformer.TypeToXmlSchema;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.ws.commons.schema.XmlSchema;
import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.model.parameter.PartParameter;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinitionBuilder;
import org.mule.connectivity.restconnect.internal.model.typesource.*;
import org.mule.connectivity.restconnect.internal.model.typesource.PrimitiveTypeSource.PrimitiveType;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil;

import javax.ws.rs.core.MediaType;
import java.io.StringWriter;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.PrimitiveMetadataModelFactory.createPrimitiveMetadataModel;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.*;


public class AMFTypeDefinitionFactory {

    private static final Logger logger = LogManager.getLogger(AMFTypeDefinitionFactory.class);


    public static TypeDefinition getTypeDefinition(Parameter parameter, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        return getTypeDefinition(parameter, null, jsonSchemaPool);
    }

    public static TypeDefinition getTypeDefinition(Parameter parameter, String mediaType, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        Shape shape = AMFParserUtil.getActualShape(parameter.schema());

        if(typeIsDefinedWithRAML(shape)) {
            return buildJsonTypeDefinitionForParameter(parameter, mediaType, jsonSchemaPool);
        }

        if(typeIsDefinedWithSchema(shape)) {
            return buildTypeDefinitionForSchemaShapeParameter(parameter, mediaType, jsonSchemaPool);
        }

        if(typeIsEnum(shape)){
            return buildJsonTypeDefinitionForParameter(parameter, mediaType, jsonSchemaPool);
        }

        if (typeIsPrimitive(shape)) {
            return buildTypeDefinitionForParameter(mediaType, parameter, createPrimitiveMetadataModel(shape));
        }

        if(shape instanceof NilShape)
            return null;

        if (typeDefinitionIsNotProvided(shape)) {
            return buildTypeDefinitionForParameter(mediaType, parameter, new PrimitiveTypeSource(PrimitiveType.STRING));
        }

        // This shouldn't happen as it's already validated that the declaration is supported.
        logger.warn("Type {} cannot be parsed.", shape.getClass());

        return null;
    }

    private static TypeDefinition buildTypeDefinitionForSchemaShapeParameter(Parameter parameter, String mediaType, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        SchemaShape schemaShape = (SchemaShape)AMFParserUtil.getActualShape(parameter.schema());

        if(isXmlSchemaShape(schemaShape)){
            return buildXmlTypeDefinitionForParameter(parameter, mediaType);
        }

        return buildJsonTypeDefinitionForParameter(parameter, mediaType, jsonSchemaPool);
    }

    private static boolean isXmlSchemaShape(SchemaShape schemaShape){
        return schemaShape.raw().nonEmpty() && schemaShape.raw().value().trim().startsWith("<");
    }

    public static TypeDefinition buildInputMetadata(amf.client.model.domain.Operation operation, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        if(operation.request() != null){
            List<Payload> payloads = operation.request().payloads();

            return getDefaultTypeDefinition(payloads, jsonSchemaPool);
        }
        return null;
    }

    public static TypeDefinition buildOutputMetadata(amf.client.model.domain.Operation operation, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        for(Response response : operation.responses()){
            if(response.statusCode().nonEmpty() && response.statusCode().value().startsWith("2")){
                return getDefaultTypeDefinition(response.payloads(), jsonSchemaPool);
            }
        }
        return null;
    }

    private static TypeDefinition getDefaultTypeDefinition(List<Payload> payloads, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        if(payloads.size() > 0){
            Optional<Payload> defaultPayload = payloads.stream().filter(x -> isDefault(x)).findFirst();

            if(defaultPayload.isPresent()){
                return getTypeDefinitionForPayload(defaultPayload.get(), jsonSchemaPool);
            }

            return getTypeDefinitionForPayload(payloads.get(0), jsonSchemaPool);
        }

        return null;
    }

    private static String getNameForPropertyShape(PropertyShape propertyShape){
        return propertyShape.id().substring(propertyShape.id().lastIndexOf("/") + 1);
    }

    private static TypeDefinition getTypeDefinitionForPayload(Payload payload, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        Shape shape = AMFParserUtil.getActualShape(payload.schema());

        if(MediaType.MULTIPART_FORM_DATA.equalsIgnoreCase(payload.mediaType().value())){
            if(shape instanceof NodeShape){
                NodeShape nodeShape = (NodeShape)shape;
                if(nodeShape.properties().size() > 0){
                    List<PartParameter> parts =
                            nodeShape.properties()
                                     .stream()
                                     .map(x -> getRestConnectPartParameter(x))
                                     .collect(Collectors.toList());

                    XmlSchema xmlSchema = TypeToXmlSchema.transform("root", shape._internal());
                    xmlSchema.setTargetNamespace("http://validationnamespace.raml.org");

                    StringWriter schemaWriter = new StringWriter();
                    xmlSchema.write(schemaWriter);
                    String schemaString = schemaWriter.toString();

                    return getTypeDefinitionBuilderForShape(MediaType.MULTIPART_FORM_DATA, shape, new MultipartTypeSource(parts,  new XmlTypeSource(schemaString)), true).build();
                }
            }
        }

        if(typeIsEnum(shape)){

            String schemaContent = ((AnyShape)shape).toJsonSchema();
            if(StringUtils.isNotBlank(schemaContent)) {
                if (jsonSchemaPool.containsSchema(schemaContent)) {
                    return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
                }
                else{
                    JsonTypeSource source = new JsonTypeSource(schemaContent);
                    TypeDefinition typeDefinition = getTypeDefinitionBuilderForShape(payload.mediaType().value(), shape, source, true).build();
                    jsonSchemaPool.putSchemaTypeDefinitionPair(schemaContent, typeDefinition);
                    return typeDefinition;
                }
            }

        }

        if(typeIsDefinedWithRAML(shape)){

            String schemaContent = ((AnyShape)shape).toJsonSchema();
            if(StringUtils.isNotBlank(schemaContent)) {
                if (jsonSchemaPool.containsSchema(schemaContent)) {
                    return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
                }
                else{
                    JsonTypeSource source = new JsonTypeSource(schemaContent);
                    TypeDefinition typeDefinition = getTypeDefinitionBuilderForShape(payload.mediaType().value(), shape, source, true).build();
                    jsonSchemaPool.putSchemaTypeDefinitionPair(schemaContent, typeDefinition);
                    return typeDefinition;
                }
            }
        }

        if(typeIsDefinedWithSchema(shape)) {
            SchemaShape schemaShape = (SchemaShape)shape;

            if(isXmlSchemaShape(schemaShape)){
                String elementName = schemaShape.annotations().fragmentName().orElse(null);
                TypeSource source = new XmlTypeSource(schemaShape.raw().value(), elementName, schemaShape.location().get());
                return getTypeDefinitionBuilderForShape(payload.mediaType().value(), shape, source, true).build();
            }
            else{
                String schemaContent = schemaShape.toJsonSchema();
                if(StringUtils.isNotBlank(schemaContent)) {
                    if (jsonSchemaPool.containsSchema(schemaContent)) {
                        return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
                    }
                    else{
                        JsonTypeSource source = new JsonTypeSource(schemaContent);
                        TypeDefinition typeDefinition = getTypeDefinitionBuilderForShape(payload.mediaType().value(), shape, source, true).build();
                        jsonSchemaPool.putSchemaTypeDefinitionPair(schemaContent, typeDefinition);
                        return typeDefinition;
                    }
                }
            }

        }

        if(typeIsPrimitive(shape)){
            return getTypeDefinitionBuilderForShape(payload.mediaType().value(), shape, createPrimitiveMetadataModel(shape), true).build();
        }

        if(shape instanceof NilShape)
            return null;

        if(typeDefinitionIsNotProvided(shape)){
            return getTypeDefinitionBuilderForShape(payload.mediaType().value(), shape, new PrimitiveTypeSource(PrimitiveType.STRING), true).build();
        }

        logger.warn("Type {} cannot be parsed.", shape.getClass());

        return null;
    }

    private static String getDefaultMediaType(Shape shape) {
        if(shape instanceof FileShape)
            return MediaType.APPLICATION_OCTET_STREAM;

        if(typeIsDefinedWithRAML(shape))
            return MediaType.APPLICATION_JSON;

        if(typeIsDefinedWithSchema(shape)){
            if(isXmlSchemaShape((SchemaShape)shape))
                return MediaType.APPLICATION_XML;
            else
                return MediaType.APPLICATION_JSON;
        }

        return MediaType.TEXT_PLAIN;
    }

    private static PartParameter getRestConnectPartParameter(PropertyShape x) {
        String name = getNameForPropertyShape(x);
        Shape range = x.range();

        String defaultMediaType = getDefaultMediaType(range);
        TypeDefinition typeDefinition = getTypeDefinitionBuilderForShape(defaultMediaType, range, new PrimitiveTypeSource(PrimitiveType.STRING), true).build();

        PartParameter part = new PartParameter(name, ParameterType.PART, typeDefinition);

        String annotatedPartNameParameter = getAnnotatedPartNameParameter(range);
        if(org.apache.commons.lang3.StringUtils.isNotBlank(annotatedPartNameParameter)){
            part.setPartNameParameterName(annotatedPartNameParameter);
        }

        String annotatedPartFilenameParameter = getAnnotatedPartFilenameParameter(range);
        if(org.apache.commons.lang3.StringUtils.isNotBlank(annotatedPartFilenameParameter)){
            part.setPartFilenameParameterName(annotatedPartFilenameParameter);
        }

        String annotatedPartName = getAnnotatedPartName(range);
        if(org.apache.commons.lang3.StringUtils.isNotBlank(annotatedPartName)){
            part.setPartName(annotatedPartName);
        }

        String annotatedPartFilename = getAnnotatedPartFilename(range);
        if(org.apache.commons.lang3.StringUtils.isNotBlank(annotatedPartFilename)){
            part.setPartFilename(annotatedPartFilename);
        }

        String annotatedPartContentType = getAnnotatedPartContentType(range);
        if(org.apache.commons.lang3.StringUtils.isNotBlank(annotatedPartContentType)){
            part.setPartContentType(annotatedPartContentType);
        }

        return part;
    }

    private static TypeDefinition buildTypeDefinitionForParameter(String mediaType, Parameter parameter, TypeSource source){
        Shape shape = AMFParserUtil.getActualShape(parameter.schema());

        return getTypeDefinitionBuilderForShape(mediaType, shape, source, parameter.required().value())
                .withDescription(parameter.description().value())
                .withAnnotatedDisplayName(AMFParserUtil.getAnnotatedParameterName(parameter))
                .build();
    }

    private static TypeDefinitionBuilder getTypeDefinitionBuilderForShape(String mediaType, Shape shape, TypeSource source, boolean required){

        TypeDefinitionBuilder builder =  new TypeDefinitionBuilder(mediaType == null ? null : MediaType.valueOf(mediaType), source, required, shape  instanceof ArrayShape, shape instanceof UnionShape);

        if(shape.defaultValueStr().nonEmpty()){
            String defaultValue = shape.defaultValueStr().value();
            defaultValue = StringEscapeUtils.escapeJava(defaultValue);

            builder.withDefaultValue(defaultValue);
        }

        if(shape instanceof AnyShape){
            AnyShape anyShape = (AnyShape)shape;
            if(!anyShape.examples().isEmpty() && anyShape.examples().get(0).value().nonEmpty()){
                builder.withExample(anyShape.examples().get(0).value().value());
            }

            if(!anyShape.values().isEmpty()){
                builder.withEnumValues(anyShape.values().stream().map(x -> getEnumValue(x)).collect(Collectors.toList()));
            }
        }

        return builder;
    }

    private static String getEnumValue(DataNode x) {
        if(x instanceof ScalarNode){
            return ((ScalarNode)x).value();
        }
        throw new IllegalArgumentException("Enum type not supported");
    }

    private static TypeDefinition buildJsonTypeDefinitionForParameter(Parameter parameter, String mediaType, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        Shape shape = AMFParserUtil.getActualShape(parameter.schema());

        try {
            AnyShape anyShape = (AnyShape)shape;

            String schemaContent = anyShape.toJsonSchema();

            if(StringUtils.isNotBlank(schemaContent)) {
                if (jsonSchemaPool.containsSchema(schemaContent)) {
                    return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
                }
                else{
                    JsonTypeSource source = new JsonTypeSource(schemaContent);
                    TypeDefinition typeDefinition = getTypeDefinitionBuilderForShape(mediaType, shape, source, parameter.required().value()).build();
                    jsonSchemaPool.putSchemaTypeDefinitionPair(schemaContent, typeDefinition);
                    return typeDefinition;
                }
            }
            else {
                return getTypeDefinitionBuilderForShape(mediaType, shape, new PrimitiveTypeSource(PrimitiveType.STRING), parameter.required().value()).build();
            }
        }

        catch (RuntimeException exception) {
            logger.warn("Couldn't generate a type definition for " + parameter.name(), exception);

            return getTypeDefinitionBuilderForShape(mediaType, shape, new PrimitiveTypeSource(PrimitiveType.STRING), parameter.required().value()).build();
        }
    }

    private static TypeDefinition buildXmlTypeDefinitionForParameter(Parameter parameter,  String mediaType) {
        Shape shape = AMFParserUtil.getActualShape(parameter.schema());

        try {
            SchemaShape schemaShape = (SchemaShape)shape;
            String elementName = schemaShape.annotations().fragmentName().orElse(null);
            XmlTypeSource source = new XmlTypeSource(schemaShape.raw().value(), elementName, schemaShape.location().get());

            TypeDefinitionBuilder builder = getTypeDefinitionBuilderForShape(mediaType, shape, source, parameter.required().value());

            return builder.build();
        }

        catch (RuntimeException exception) {
            logger.warn("Couldn't generate a type definition for " + parameter.name(), exception);

            return getTypeDefinitionBuilderForShape(mediaType, shape, new PrimitiveTypeSource(PrimitiveType.STRING), parameter.required().value()).build();
        }
    }

    public static boolean typeIsDefinedWithSchema(Shape shape){
        return shape instanceof SchemaShape;
    }

    private static boolean typeIsEnum(Shape shape) {
        return shape instanceof AnyShape && !shape.values().isEmpty();
    }

    public static boolean typeIsDefinedWithRAML(Shape shape) {
        return shape instanceof NodeShape ||
               shape instanceof ArrayShape ||
               shape instanceof UnionShape;
    }

    public static boolean typeIsPrimitive(Shape shape) {
        return shape instanceof ScalarShape ||
                shape instanceof FileShape;
    }

    // This is for the case that the type is not defined (for instance, when there is an example but not a definition).
    public static boolean typeDefinitionIsNotProvided(Shape shape) {
        return shape instanceof AnyShape && !typeIsDefinedWithRAML(shape) && !typeIsPrimitive(shape)&& !typeIsDefinedWithSchema(shape);
    }

}
