package org.mule.connectivity.restconnect.internal.modelGeneration.amf.security;

import amf.client.model.StrField;
import amf.client.model.domain.*;
import edu.emory.mathcs.backport.java.util.Arrays;
import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.exception.UnsupportedSecuritySchemeException;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.model.security.*;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.common.security.SecuritySchemeFactory;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

import static org.mule.connectivity.restconnect.internal.model.type.TypeDefinitionBuilder.buildSimpleStringType;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.getParameterList;

public class AMFSecuritySchemeFactory {

    public static List<APISecurityScheme> getSchemesForOperation(EndPoint endPoint, Operation operation, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<SecurityScheme> globalSchemes = endPoint.security().stream().map(x -> getScheme(x)).collect(Collectors.toList());

        List<SecurityScheme> endPointSchemes = endPoint.security().stream().map(x -> getScheme(x)).collect(Collectors.toList());

        List<SecurityScheme> operationSchemes = operation.security().stream().map(x -> getScheme(x)).collect(Collectors.toList());

        return getOperationSecuritySchemes(operationSchemes, endPointSchemes, globalSchemes, jsonSchemaPool);
    }

    @SuppressWarnings("Duplicates")
    private static List<APISecurityScheme> getOperationSecuritySchemes(List<SecurityScheme> operationSchemes, List<SecurityScheme> endPointSchemes, List<SecurityScheme> globalSchemes, JsonSchemaPool jsonSchemaPool) throws GenerationException {

        List<SecurityScheme> securitySchemesForOperation = SecuritySchemeFactory.getSecuritySchemesForOperation(operationSchemes, endPointSchemes, globalSchemes);

        if(!securitySchemesForOperation.isEmpty()){
            return getAPISecuritySchemes(securitySchemesForOperation, jsonSchemaPool);
        }
        else{
            //If there are no security schemes defined for this operation, we create an UnsecuredScheme
            List<APISecurityScheme> returnSchemes = new LinkedList<>();
            returnSchemes.add(new UnsecuredScheme());
            return returnSchemes;
        }
    }

    @SuppressWarnings("Duplicates")
    private static List<APISecurityScheme> getAPISecuritySchemes(List<SecurityScheme> sourceSchemes, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<APISecurityScheme> returnSchemes = new LinkedList<>();

        for (SecurityScheme scheme : sourceSchemes) {
            List<APISecurityScheme> generatedSchemes = createSecuritySchemes(scheme, jsonSchemaPool);

            for(APISecurityScheme generatedScheme : generatedSchemes){
                if(returnSchemes.stream().noneMatch(x -> x.equals(generatedScheme))){
                    returnSchemes.add(generatedScheme);
                }
            }
        }

        // If the API is secured but we don't support any defined schemes, then we must throw a generation exception.
        if(!sourceSchemes.isEmpty() && returnSchemes.isEmpty()){
            throw new UnsupportedSecuritySchemeException("None of the specified security schemes ( " + listSchemes(sourceSchemes) + ") are supported.");
        }

        return returnSchemes;
    }

    private static List<APISecurityScheme> createSecuritySchemes(SecurityScheme scheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        ArrayList<APISecurityScheme> apiSecuritySchemes = new ArrayList<>();

        if (scheme == null || !scheme.type().nonEmpty()){
            apiSecuritySchemes.add(new UnsecuredScheme());
            return apiSecuritySchemes;
        }

        String schemeType = scheme.type().value();

        if(AMFSecuritySchemesNaming.isBasicAuth(schemeType)){
            apiSecuritySchemes.add(new BasicAuthScheme());
        }
        else if(AMFSecuritySchemesNaming.isPassThrough(schemeType)){
            apiSecuritySchemes.add(buildPassThroughSecurityScheme(scheme, jsonSchemaPool));
        }
        else if(AMFSecuritySchemesNaming.isApiKey(schemeType)){
            apiSecuritySchemes.add(buildApiKeySecurityScheme(scheme, jsonSchemaPool));
        }
        else if(AMFSecuritySchemesNaming.isOauth2(schemeType)){
            OAuth2Settings oAuth2Settings = (OAuth2Settings)scheme.settings();

            if (AMFOauth2FlowsNaming.isAuthorizationCode(oAuth2Settings)) {
                apiSecuritySchemes.add(buildOAuth2AuthorizationCodeSecurityScheme(oAuth2Settings));
            } else if (AMFOauth2FlowsNaming.isClientCredentials(oAuth2Settings)) {
                apiSecuritySchemes.add(buildOAuth2ClientCredentialsSecurityScheme(oAuth2Settings));
            }
        }
        else if(AMFSecuritySchemesNaming.isDigestAuth(schemeType)){
            apiSecuritySchemes.add(new DigestAuthenticationScheme());
        }
        else if (AMFSecuritySchemesNaming.isJwtAuth(schemeType)) {
            apiSecuritySchemes.add(buildJwtAuthenticationScheme(scheme, jsonSchemaPool));
        }
        else if(AMFSecuritySchemesNaming.isCustom(schemeType)){
            apiSecuritySchemes.add(buildCustomAuthenticationScheme(scheme, jsonSchemaPool));
        }

        return apiSecuritySchemes;
    }

    private static SecurityScheme getScheme(DomainElement domainElement) {
        if(domainElement instanceof ParametrizedSecurityScheme){
            ParametrizedSecurityScheme securityScheme = (ParametrizedSecurityScheme)domainElement;
            if(securityScheme.scheme() == null){
                return null;
            }
            if(securityScheme.scheme().isLink() && securityScheme.scheme().linkTarget() != null){
                return (SecurityScheme)securityScheme.scheme().linkTarget().get();
            }
            else{
                return securityScheme.scheme();
            }
        }
        else{
            return (SecurityScheme)domainElement;
        }
    }

    private static PassThroughScheme buildPassThroughSecurityScheme(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Parameter> queryParameters = getParameterList(securityScheme.queryParameters(), ParameterType.SECURITY, jsonSchemaPool);
        List<Parameter> headers = getParameterList(securityScheme.headers(), ParameterType.SECURITY, jsonSchemaPool);

        return new PassThroughScheme(queryParameters, headers);
    }

    private static PassThroughScheme buildApiKeySecurityScheme(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Parameter> queryParameters = getParameterList(securityScheme.queryParameters(), ParameterType.SECURITY, jsonSchemaPool);
        List<Parameter> headers = getParameterList(securityScheme.headers(), ParameterType.SECURITY, jsonSchemaPool);

        if(securityScheme.settings() instanceof ApiKeySettings){
            String in = ((ApiKeySettings)securityScheme.settings()).in().value();
            String name = ((ApiKeySettings)securityScheme.settings()).name().value();

            if(in.equalsIgnoreCase("query")){
                queryParameters.add(new Parameter(name, ParameterType.SECURITY, buildSimpleStringType(true)));
            }
            else{
                headers.add(new Parameter(name, ParameterType.SECURITY, buildSimpleStringType(true)));
            }
        }

        return new PassThroughScheme(queryParameters, headers);
    }

    //TODO AMF does not support JWT parser to get the security scheme information.
    //     Opened a bug to https://www.mulesoft.org/jira/browse/APIMF-1374
    private static JwtAuthenticationScheme buildJwtAuthenticationScheme(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Parameter> queryParameters = getParameterList(securityScheme.queryParameters(), ParameterType.SECURITY, jsonSchemaPool);
        List<Parameter> headers = getParameterList(securityScheme.headers(), ParameterType.SECURITY, jsonSchemaPool);
        List<Parameter> body = new LinkedList<>();
        body.addAll(queryParameters);
        body.addAll(headers);

        return new JwtAuthenticationScheme(headers, body, queryParameters);
    }

    private static CustomAuthenticationScheme buildCustomAuthenticationScheme(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Parameter> queryParameters = getParameterList(securityScheme.queryParameters(), ParameterType.SECURITY, jsonSchemaPool);
        List<Parameter> headers = getParameterList(securityScheme.headers(), ParameterType.SECURITY, jsonSchemaPool);

        return new CustomAuthenticationScheme(queryParameters, headers);
    }

    private static OAuth2AuthorizationCodeScheme buildOAuth2AuthorizationCodeSecurityScheme(OAuth2Settings settings){
        String authorizationUri = settings.authorizationUri().value();
        String accessTokenUri = settings.accessTokenUri().value();
        List<String> authorizationGrants = Arrays.asList(new String[]{ OAuth2Scheme.OAUTH2_GRANT_AUTHORIZATION_CODE });
        List<String> scopes = settings.scopes().stream().map(s -> s.name().value()).collect(Collectors.toList());

        return new OAuth2AuthorizationCodeScheme(authorizationUri, accessTokenUri, authorizationGrants, scopes);
    }

    private static OAuth2ClientCredentialsScheme buildOAuth2ClientCredentialsSecurityScheme(OAuth2Settings settings){
        String authorizationUri = settings.authorizationUri().value();
        String accessTokenUri = settings.accessTokenUri().value();
        List<String> authorizationGrants = Arrays.asList(new String[]{ OAuth2Scheme.OAUTH2_GRANT_CLIENT_CREDENTIALS });
        List<String> scopes = settings.scopes().stream().map(s -> s.name().value()).collect(Collectors.toList());

        return new OAuth2ClientCredentialsScheme(authorizationUri, accessTokenUri, authorizationGrants, scopes);
    }

    private static String listSchemes(List<SecurityScheme> securitySchemes){
        StringBuilder builder = new StringBuilder();
        for(SecurityScheme securityScheme : securitySchemes){
            builder.append(securityScheme.name());
            builder.append(":");
            builder.append(securityScheme.type().value());
            if(securityScheme.type().value().equals(APISecurityScheme.OAUTH2) && securityScheme.settings() != null){
                OAuth2Settings settings = (OAuth2Settings)securityScheme.settings();
                builder.append(" :");
                if(settings.authorizationGrants() != null && settings.authorizationGrants().size() > 0){
                    for(StrField grant : settings.authorizationGrants()){
                        builder.append(" ");
                        builder.append(grant.value());
                    }
                }
            }
            builder.append(". ");
        }

        return builder.toString();
    }

}
