package org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util;

import org.apache.commons.lang3.StringUtils;
import org.mule.connectivity.restconnect.internal.model.HTTPMethod;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.model.parameter.PartParameter;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils;
import org.raml.v2.api.model.v10.api.Api;
import org.raml.v2.api.model.v10.common.Annotable;
import org.raml.v2.api.model.v10.datamodel.FileTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.TypeDeclaration;
import org.raml.v2.api.model.v10.declarations.AnnotationRef;
import org.raml.v2.api.model.v10.methods.Method;
import org.raml.v2.api.model.v10.system.types.AnnotableStringType;

import javax.ws.rs.core.MediaType;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

import static org.apache.commons.lang3.StringUtils.trimToNull;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.RamlParserTypeDefinitionFactory.*;


public class RamlParserUtils extends ParserUtils {

    // Overrides default construction to avoiding instantiation.
    protected RamlParserUtils() {

    }

    public static String getMethodDescription(Method method) {
        String description = null;
        if(method.description() != null)
            description = method.description().value();

        else if(method.resource() != null && method.resource().description() != null)
            description = method.resource().description().value();

        return trimToNull(description);
    }

    public static String getMethodDisplayName(Method method) {
        String displayName = null;
        // This checks for the case that the displayName attribute isn't provided, as the parsers
        // defaults this to the method name, per the RAML 1.0 spec.
        if(method.displayName() != null && method.displayName().value() != null && !method.displayName().value().equals(method.method()))
            displayName = method.displayName().value();

        return trimToNull(displayName);
    }


    public static String getCanonicalOperationName(Method method, String friendlyName) {
        String annotatedOperationName = getAnnotatedOperationName(method);

        HTTPMethod httpMethod = HTTPMethod.fromString(method.method());

        return getCanonicalOperationName(httpMethod, method.resource().resourcePath(), friendlyName, annotatedOperationName);
    }

    public static String getAnnotatedOperationName(Method method) {
        return getAnnotationStringValue(method, OPERATION_NAME_ANNOTATION);
    }

    public static String getAnnotatedParameterName(TypeDeclaration typeDeclaration) {
        return getAnnotationStringValue(typeDeclaration, PARAMETER_NAME_ANNOTATION);
    }

    public static String getAnnotatedPartName(TypeDeclaration typeDeclaration) {
        return getAnnotationStringValue(typeDeclaration, PART_NAME_ANNOTATION);
    }

    public static String getAnnotatedPartFilename(TypeDeclaration typeDeclaration) {
        return getAnnotationStringValue(typeDeclaration, PART_FILENAME_ANNOTATION);
    }

    public static String getAnnotatedPartContentType(TypeDeclaration typeDeclaration) {
        return getAnnotationStringValue(typeDeclaration, PART_CONTENT_TYPE_ANNOTATION);
    }

    public static String getAnnotatedPartNameParameter(TypeDeclaration typeDeclaration) {
        return getAnnotationStringValue(typeDeclaration, PART_NAME_PARAMETER_ANNOTATION);
    }

    public static String getAnnotatedPartFilenameParameter(TypeDeclaration typeDeclaration) {
        return getAnnotationStringValue(typeDeclaration, PART_FILENAME_PARAMETER_ANNOTATION);
    }

    public static String getAnnotationStringValue(Annotable target, String annotationName){
        List<AnnotationRef> annotations = target.annotations();
        if(annotations != null && !annotations.isEmpty()){
            Optional<AnnotationRef> annotation = annotations.stream()
                                                  .filter(x -> x.annotation().name().equalsIgnoreCase(annotationName))
                                                  .findFirst();

            if(annotation.isPresent()){
                return (String)annotation.get().structuredValue().value();
            }
        }

        return null;
    }

    public static List<Parameter> getParameterList(List<TypeDeclaration> typeDeclarationParameters, ParameterType parameterType, JsonSchemaPool jsonSchemaPool) throws Exception {
        return getParameterList(typeDeclarationParameters, false, parameterType, jsonSchemaPool);
    }

    public static List<Parameter> getParameterList(List<TypeDeclaration> typeDeclarationParameters, boolean setDefaultMediaTypes, ParameterType parameterType, JsonSchemaPool jsonSchemaPool) throws Exception {
        List<Parameter> parameters = new LinkedList<>();

        for (TypeDeclaration typeDeclaration : typeDeclarationParameters) {
            if(setDefaultMediaTypes){
                parameters.add(new Parameter(typeDeclaration.name(), parameterType, getTypeDefinition(typeDeclaration, getDefaultMediaType(typeDeclaration), jsonSchemaPool)));
            }
            else{
                parameters.add(new Parameter(typeDeclaration.name(), parameterType, getTypeDefinition(typeDeclaration, jsonSchemaPool)));
            }
        }

        return parameters;
    }

    public static List<PartParameter> getPartParameterList(List<TypeDeclaration> typeDeclarationParameters, JsonSchemaPool jsonSchemaPool) throws Exception {
        List<PartParameter> partParameters = new LinkedList<>();

        for (TypeDeclaration typeDeclaration : typeDeclarationParameters) {
            PartParameter part = new PartParameter(typeDeclaration.name(), ParameterType.PART, getTypeDefinition(typeDeclaration, getDefaultMediaType(typeDeclaration), jsonSchemaPool));

            String annotatedPartNameParameter = getAnnotatedPartNameParameter(typeDeclaration);
            if(StringUtils.isNotBlank(annotatedPartNameParameter)){
                part.setPartNameParameterName(annotatedPartNameParameter);
            }

            String annotatedPartFilenameParameter = getAnnotatedPartFilenameParameter(typeDeclaration);
            if(StringUtils.isNotBlank(annotatedPartFilenameParameter)){
                part.setPartFilenameParameterName(annotatedPartFilenameParameter);
            }

            String annotatedPartName = getAnnotatedPartName(typeDeclaration);
            if(StringUtils.isNotBlank(annotatedPartName)){
                part.setPartName(annotatedPartName);
            }

            String annotatedPartFilename = getAnnotatedPartFilename(typeDeclaration);
            if(StringUtils.isNotBlank(annotatedPartFilename)){
                part.setPartFilename(annotatedPartFilename);
            }

            String annotatedPartContentType = getAnnotatedPartContentType(typeDeclaration);
            if(StringUtils.isNotBlank(annotatedPartContentType)){
                part.setPartContentType(annotatedPartContentType);
            }

            partParameters.add(part);
        }

        return partParameters;
    }

    private static String getDefaultMediaType(TypeDeclaration typeDeclaration) {
        if(typeDeclaration instanceof FileTypeDeclaration)
            return MediaType.APPLICATION_OCTET_STREAM;

        if(typeIsDefinedWithRAML(typeDeclaration))
            return MediaType.APPLICATION_JSON;

        if(typeIsDefinedWithJSONSchema(typeDeclaration))
            return MediaType.APPLICATION_JSON;

        if(typeIsDefinedWithXMLSchema(typeDeclaration))
            return MediaType.APPLICATION_XML;

        return MediaType.TEXT_PLAIN;
    }

    public static String getValueFromAnnotableString(AnnotableStringType annotableString) {
        return annotableString == null ? null : annotableString.value();
    }

    public static boolean isIgnored(Method method) {
        for (AnnotationRef annotationRef : method.annotations()) {
            if(IGNORE_ANNOTATION.equals(annotationRef.annotation().name()))
                return true;
        }

        return false;
    }

    public static boolean isDefault(TypeDeclaration typeDeclaration) {
        for (AnnotationRef annotationRef : typeDeclaration.annotations()) {
            if(DEFAULT_ANNOTATION.equals(annotationRef.annotation().name()))
                return true;
        }

        return false;
    }

    public static boolean isGenerateUserSelectedSecuritySchemes(Method method, Api api) {
        for (AnnotationRef annotationRef : method.annotations()) {
            if(USER_SELECTED_SECURITY_SCHEMES_ANNOTATION.equals(annotationRef.annotation().name()))
                return true;
        }

        for (AnnotationRef annotationRef : api.annotations()) {
            if(USER_SELECTED_SECURITY_SCHEMES_ANNOTATION.equals(annotationRef.annotation().name()))
                return true;
        }

        return false;
    }
}
