package org.mule.connectivity.restconnect.internal.modelGeneration.util;

import org.apache.commons.lang3.StringUtils;
import org.mule.connectivity.restconnect.internal.model.HTTPMethod;
import scala.Char;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static java.util.Collections.reverse;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.join;
import static org.raml.v2.internal.utils.Inflector.*;


public class ParserUtils {

    public static final String IGNORE_ANNOTATION = "ignored";
    public static final String DEFAULT_ANNOTATION = "default";
    public static final String USER_SELECTED_SECURITY_SCHEMES_ANNOTATION = "userSelectedSecuritySchemes";
    public static final String PARAMETER_NAME_ANNOTATION = "parameterName";
    public static final String OPERATION_NAME_ANNOTATION = "operationName";
    public static final String PART_NAME_ANNOTATION = "partName";
    public static final String PART_FILENAME_ANNOTATION = "partFilename";
    public static final String PART_CONTENT_TYPE_ANNOTATION = "partContentType";
    public static final String PART_NAME_PARAMETER_ANNOTATION = "partNameParameter";
    public static final String PART_FILENAME_PARAMETER_ANNOTATION = "partFilenameParameter";

    // Overrides default construction to avoiding instantiation.
    protected ParserUtils() {

    }

    public static String removeJavaNameUnwantedCharacters(String name) {
        return removeJavaNameUnwantedCharacters(name, "j");
    }

    public static String removeJavaNameUnwantedCharacters(String name, String sanitizeSufix) {
        if(name == null)
            return null;

        String javaName = name.replaceAll("[^a-zA-Z0-9_]", "");

        if(javaName.matches("^[0-9_].*") || !isValidDevKitParameterName(javaName)){
            javaName = sanitizeSufix + javaName;
        }

        if(StringUtils.isBlank(javaName)){
            return "empty";
        }

        return javaName;
    }

    public static String removeJavaPackageUnwantedCharacters(String name) {
        if(name == null)
            return null;

        String[] split = name.split("[.]");

        StringBuilder packageString = new StringBuilder();
        for(int i = 0; i < split.length; i++){
            packageString.append(removeJavaNameUnwantedCharacters(split[i]).toLowerCase());
            if(i < split.length - 1){
                packageString.append(".");
            }
        }

        return packageString.toString();
    }

    public static String removeMavenArtifactUnwantedCharacters(String name) {
        if(name == null)
            return null;

        return name.replaceAll("[^a-zA-Z0-9_\\-]", "");
    }

    public static String removeMavenGroupUnwantedCharacters(String name) {
        if(name == null)
            return null;

        return name.replaceAll("[^a-zA-Z0-9\\._\\-]", "");
    }

    public static String removeMavenVersionUnwantedCharacters(String name) {
        if(name == null)
            return null;

        return name.replaceAll("[^a-zA-Z0-9\\._\\-]", "");
    }

    public static String removeXmlUnwantedCharacters(String name) {
        if(name == null)
            return null;

        return name.replaceAll("[^a-zA-Z0-9_-]", "");
    }

    public static String getCapUnderscoreName(String name) {
        return splitCapsWithUnderscores(getSanitizedOperationName(name)).toUpperCase();
    }

    public static String getSanitizedOperationName(String name) {
        name = name.replaceAll("-", "_");
        name = name.replaceAll("[^a-zA-Z0-9_-]", "");

        if(name.startsWith("_"))
            name = name.replaceFirst("_", "");

        return name;
    }

    public static String getXmlName(String name) {
        return splitCapsWithHypens(getXmlSanitizedName(name)).toLowerCase();
    }

    public static String splitCapsWithUnderscores(String name){
        return splitCaps(name, "_");
    }

    public static String splitCapsWithHypens(String name){
        return splitCaps(name, "-");
    }

    public static String splitCaps(String name, String split){
        StringBuilder builder = new StringBuilder();

        for (int i = 0; i < name.length(); i ++){
            if(i + 1 < name.length()
                    && Character.isLowerCase(name.charAt(i))
                    && Character.isUpperCase(name.charAt(i + 1))){
                builder.append(name.charAt(i));
                builder.append(split);
            }
            else if(i + 1 < name.length()
                    && i - 1 >= 0
                    && Character.isUpperCase(name.charAt(i))
                    && Character.isLowerCase(name.charAt(i + 1))
                    && Character.isUpperCase(name.charAt(i - 1))
            ){
                builder.append(split);
                builder.append(name.charAt(i));
            }
            else if(i + 1 < name.length()
                    && i > 0
                    && (name.charAt(i) == '_' || String.valueOf(name.charAt(i)).equals(split))
                    && Character.isAlphabetic(name.charAt(i + 1))){
                builder.append(split);
            }
            else{
                builder.append(name.charAt(i));
            }
        }

        return builder.toString();
    }

    public static String splitNumbersAndLetters(String name, String split){
        StringBuilder builder = new StringBuilder();

        for (int i = 0; i < name.length(); i ++){
            if(i + 1 < name.length() && Character.isAlphabetic(name.charAt(i)) && Character.isDigit(name.charAt(i+1))){
                builder.append(name.charAt(i));
                builder.append(split);
            }
            else if(i + 1 < name.length() && Character.isDigit(name.charAt(i)) && Character.isAlphabetic(name.charAt(i+1))){
                builder.append(name.charAt(i));
                builder.append(split);
            }
            else{
                builder.append(name.charAt(i));
            }
        }

        return builder.toString();
    }

    public static String getXmlSanitizedName(String name) {
        name = removeXmlUnwantedCharacters(name);

        if(StringUtils.isEmpty(name))
            return "";

        if(name.toLowerCase().startsWith("xml") || Character.isDigit(name.charAt(0)))
            name = "_" + name;

        if(name.startsWith("-"))
            name = name.replaceFirst("-", "_");

        return name;
    }

    public static String getJavaName(String operationName) {
        return lowercamelcase(removeJavaNameUnwantedCharacters(operationName));
    }

    protected static String getCanonicalNameFromFriendlyName(String friendlyName) {
        // Removes all sentence stuff.
        friendlyName = friendlyName.replace(" ", "_");
        return removeXmlUnwantedCharacters(friendlyName.toUpperCase());
    }

    protected static String getParameterName(String uriValue) {
        String cleanParameterName = uriValue.replace("{", "").replace("}", "");
        return upperunderscorecase(cleanParameterName);
    }

    protected static boolean hasUriParameter(String uriValue) {
        return uriValue.matches("\\{.*\\}");
    }



    public static String removeSpaces(String string) {
        return string == null ? null : string.replace(" ", "");
    }

    public static String removeDots(String string) {
        return string == null ? null : string.replace(".", "");
    }

    public static String removeHyphens(String string) {
        return string == null ? null : string.replace("-", "");
    }

    public static String removeSpacesAndHyphens(String string) {
        return removeHyphens(removeSpaces(string));
    }

    private static final List<String> reservedJavaWords = Arrays.asList(
            "abstract",  "assert",       "boolean",    "break",      "byte",      "case",
            "catch",     "char",         "class",      "const",     "continue",
            "default",   "do",           "double",     "else",      "extends",
            "false",     "final",        "finally",    "float",     "for",
            "goto",      "if",           "implements", "import",    "instanceof",
            "int",       "interface",    "long",       "native",    "new",
            "null",      "package",      "private",    "protected", "public",
            "return",    "short",        "static",     "strictfp",  "super",
            "switch",    "synchronized", "this",       "throw",     "throws",
            "transient", "true",         "try",        "void",      "volatile",
            "while"
    );

    public static boolean isReservedJavaWord(String word){
        return reservedJavaWords.contains(word);
    }

    private static final List<String> reservedProcessorParameterNames = Arrays.asList(
            "name",     "friendlyName",     "operationName"
    );

    public static boolean isReservedProcessorVariableName(String word){
        return reservedProcessorParameterNames.contains(word);
    }

    public static boolean isValidDevKitParameterName(String word){
        return !isReservedJavaWord(word) && !isReservedProcessorVariableName(word);
    }

    protected static String getCanonicalOperationName(HTTPMethod method, String path, String friendlyName, String annotatedName){
        // This method produces an intermediate operation name consisting of uppercase words separated by underscores.

        // First tries to use the operationName annotation if it exists.
        if(annotatedName != null)
            return getCanonicalNameFromFriendlyName(annotatedName);

        // If the display name was provided, then we use that.
        if(friendlyName != null)
            return getCanonicalNameFromFriendlyName(friendlyName);

        // Otherwise, we need to produce it ourselves.
        return buildCanonicalOperationName(method, path);
    }

    private static String replaceOdataParams(String string){
        Pattern oDataParameterPattern = Pattern.compile("([^\\s\\(\\)\\{\\}\\=\\,]+)\\=\\'\\{([^\\s\\(\\)\\{\\}\\=\\,]+)\\}\\'");
        Matcher oDataParameterMatcher = oDataParameterPattern.matcher(string);
        return oDataParameterMatcher.replaceAll("$1");
    }

    private static String buildCanonicalOperationName(HTTPMethod method, String path) {
        // Builds the operation name according to the method and path.
        //      GET     /towns                               ->  getTowns
        //      GET     /towns/{id}                          ->  getTownById
        //      POST    /towns                               ->  createTown
        //      PUT     /towns/{id}/streets                  ->  updateStreetById
        //      DELETE  /towns/{townId}/streets/{streetId}   ->  deleteStreetByTownIdStreetId
        // Note that:
        //      - Only the last resource is listed (deleteStreet...)
        //      - All URI params are listed, from left to right (...ByTownIdStreetId)
        StringBuilder operationNameBuilder = new StringBuilder().append(method.getVerbName());

        // This holds the last resource.
        String resourceName = null;
        // We use this variable to indicate if the resource is a collection or an element for pluralization at the end.
        Boolean isElement = null;
        // This keeps all parameters.
        List<String> parameters = new ArrayList<>();

        // Collects the names of the remaining parameters (if any) and saves the resource name (if not found yet).
        String remainingPath = replaceOdataParams(path);
        do {
            String pathSegment = remainingPath.substring(remainingPath.lastIndexOf("/") + 1);

            // Updates the isElement variable according to the last resource of the segment.
            boolean hasParameter = hasUriParameter(pathSegment);
            if(isElement == null)
                isElement = hasParameter;

            // Appends the parameter to the list.
            if(hasParameter)
                parameters.add(getCapUnderscoreName(pathSegment));

                // Note that if it is a parameter then it can't be a resource. This updates the resource.
            else if(resourceName == null)
                resourceName = pathSegment;

            remainingPath = remainingPath.substring(0, remainingPath.lastIndexOf("/"));
        } while (isNotBlank(remainingPath) && !remainingPath.equals("/"));

        // This handles the case of an endpoint defined in "/".
        if(isNotBlank(resourceName)) {
            // Converts the resource to singular / plural.
            if(isElement || method.isAlwaysSingular())
                resourceName = singularize(resourceName);
            else
                resourceName = pluralize(resourceName);

            operationNameBuilder.append("_").append(getCapUnderscoreName(resourceName));
        }

        // Appends all the parameters (in reverse order, as it should go from left to right).
        if(!parameters.isEmpty()) {
            reverse(parameters);
            operationNameBuilder.append("_BY_").append(join(parameters, "_"));
        }

        return operationNameBuilder.toString();
    }
}
