package org.mule.connectivity.restconnect.internal.templateEngine.decorator.model.sdk;

import org.apache.velocity.VelocityContext;
import org.mule.connectivity.restconnect.internal.model.api.RestConnectAPIModel;
import org.mule.connectivity.restconnect.internal.model.security.APISecurityScheme;

import java.nio.file.Path;
import java.util.Map;
import java.util.function.Function;

import static com.google.common.base.CaseFormat.LOWER_UNDERSCORE;
import static com.google.common.base.CaseFormat.UPPER_CAMEL;
import static java.lang.String.format;
import static java.util.stream.Collectors.toMap;
import static org.apache.commons.lang.WordUtils.capitalizeFully;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils.getXmlName;

public class ConnectionProvider extends ExtensionEntity {
    private final String name;
    private final String extensionName;
    private final String displayName;
    private final String xml;
    private final String endpointUrl;
    private final Map<String, Parameter> headers;
    private final Map<String, Parameter> parameters;
    private final Map<String, Parameter> requestParameters;

    public ConnectionProvider(RestConnectAPIModel model, APISecurityScheme scheme, Path sourcesPath, Path resourcesPath, VelocityContext context) {
        super(sourcesPath, resourcesPath, context);
        this.extensionName = capitalizeFully(getXmlName(model.getApiName()).replace("-connector", ""));
        this.name = scheme.getSchemeName().replace(" ", "").replace(".", "");
        this.displayName = scheme.getSchemeName();
        this.xml = UPPER_CAMEL.to(LOWER_UNDERSCORE, displayName.toLowerCase().replace(" ", "-").replace(".", ""));
        this.endpointUrl = model.getBaseUri() != null ? model.getBaseUri().toString() : "baseuri.com";
        this.headers = scheme.getHeaders().stream()
                .map(parameter -> new Parameter(parameter.getInternalName(), parameter.getTypeDefinition(), sourcesPath, resourcesPath, context))
                .collect(toMap(Parameter::getName, Function.identity()));
        this.parameters = scheme.getConfigParameters().stream()
                .map(parameter -> new Parameter(parameter.getInternalName(), parameter.getTypeDefinition(), sourcesPath, resourcesPath, context))
                .collect(toMap(Parameter::getName, Function.identity()));
        this.requestParameters = scheme.getQueryParameters().stream()
                .map(parameter -> new Parameter(parameter.getInternalName(), parameter.getTypeDefinition(), sourcesPath, resourcesPath, context))
                .collect(toMap(Parameter::getName, Function.identity()));

    }

    public String getName() {
        return name;
    }

    public String getDisplayName() {
        return displayName;
    }

    public String getXml() {
        return xml;
    }

    public String getEndpointUrl() {
        return endpointUrl;
    }

    public Map<String, Parameter> getHeaders() {
        return headers;
    }

    public Map<String, Parameter> getParameters() {
        return parameters;
    }

    public Map<String, Parameter> getRequestParameters() {
        return requestParameters;
    }

    @Override
    public void generate() {
        applyTemplate(format("templates/sdk/connection/provider/%sConnectionProvider.vm", name), format("internal/connection/provider/%sConnectionProvider.java", name));

        if(name.equalsIgnoreCase(APISecurityScheme.JWT)) {
            applyTemplate("templates/sdk/connection/provider/param/jwt/AccessTokenRequestParameterGroup.vm", format("internal/connection/provider/param/jwt/%sAccessTokenRequestParameterGroup.java", extensionName));
            applyTemplate("templates/sdk/connection/provider/param/jwt/JwtClaimsParameterGroup.vm", format("internal/connection/provider/param/jwt/%sJwtClaimsParameterGroup.java", extensionName));
            applyTemplate("templates/sdk/connection/provider/param/jwt/JwtHeadersParameterGroup.vm", format("internal/connection/provider/param/jwt/%sJwtHeadersParameterGroup.java", extensionName));
        }

        headers.values().stream().forEach(Parameter::generate);
        parameters.values().stream().forEach(Parameter::generate);
    }
}
