package org.mule.connectivity.restconnect.internal.templateEngine.decorator.operation;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang.StringUtils;
import org.jsonschema2pojo.rules.RuleFactory;
import org.apache.commons.lang3.StringEscapeUtils;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.security.APISecurityScheme;
import org.mule.connectivity.restconnect.internal.templateEngine.PojoFQNamePool;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.security.devkit.DevKitSecuritySchemeDecorator;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.security.devkit.DevKitSecuritySchemeDecoratorFactory;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.type.DevKitConnectorTypeDefinitionDecorator;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.apache.commons.lang.WordUtils.capitalize;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils.*;


public class DevKitOperationDecorator extends OperationDecorator {

    private final String basePackage;
    private final DevKitConnectorTypeDefinitionDecorator inputMetadata;
    private final DevKitConnectorTypeDefinitionDecorator outputMetadata;
    private final List<DevKitConnectorTypeDefinitionDecorator> queryParameters;
    private final List<DevKitConnectorTypeDefinitionDecorator> uriParameters;
    private final List<DevKitConnectorTypeDefinitionDecorator> parameters;
    private final List<DevKitConnectorTypeDefinitionDecorator> headers;
    private final List<DevKitConnectorTypeDefinitionDecorator> partParameters;
    private final List<DevKitSecuritySchemeDecorator> securitySchemes;
    private final Map<APISecurityScheme, String> securitySchemesNames;
    private final PojoFQNamePool pojoFQNamePool;

    public DevKitOperationDecorator(Operation operation, String basePackage, Map<APISecurityScheme, String> securitySchemesNames, PojoFQNamePool pojoFQNamePool) {
        super(operation);
        this.securitySchemesNames = securitySchemesNames;
        this.pojoFQNamePool = pojoFQNamePool;
        this.basePackage = basePackage;
        this.inputMetadata = buildInputMetadata();
        this.outputMetadata = buildOutputMetadata();
        this.queryParameters = buildDecoratedQueryParameters();
        this.uriParameters = buildDecoratedUriParameters();
        this.headers =  buildDecoratedHeaders();
        this.partParameters = buildDecoratedPartParameters();
        this.parameters = buildDecoratedParameters();
        this.securitySchemes = buildDecoratedSecuritySchemes();
    }

    private DevKitConnectorTypeDefinitionDecorator buildInputMetadata() {
        if(this.getInputMetadata() == null)
            return null;

        DevKitConnectorTypeDefinitionDecorator inputMetadata = new DevKitConnectorTypeDefinitionDecorator(this.getInputMetadata(), basePackage, pojoFQNamePool);
        inputMetadata.setClassName(capitalize(getMethodName()) + "Request");
        return inputMetadata;
    }

    private DevKitConnectorTypeDefinitionDecorator buildOutputMetadata() {
        if(this.getOutputMetadata() == null)
            return null;

        DevKitConnectorTypeDefinitionDecorator outputMetadata = new DevKitConnectorTypeDefinitionDecorator(this.getOutputMetadata(), basePackage, pojoFQNamePool);
        outputMetadata.setClassName(capitalize(getMethodName()) + "Response");
        return outputMetadata;
    }

    private List<DevKitConnectorTypeDefinitionDecorator> buildDecoratedPartParameters() {
        return buildParameters(this.getPartParameters());
    }

    private List<DevKitConnectorTypeDefinitionDecorator> buildDecoratedUriParameters() {
        return buildParameters(this.getUriParameters());
    }

    private List<DevKitConnectorTypeDefinitionDecorator> buildDecoratedQueryParameters() {
        return buildParameters(this.getQueryParameters());
    }

    private List<DevKitConnectorTypeDefinitionDecorator> buildDecoratedHeaders() {
        return buildParameters(this.getHeaders());
    }

    private List<DevKitSecuritySchemeDecorator> buildDecoratedSecuritySchemes() {
        return operation.getSecuritySchemes()
                .stream()
                .map(x -> DevKitSecuritySchemeDecoratorFactory.getSecuritySchemeDecorator(x, securitySchemesNames, pojoFQNamePool))
                .collect(Collectors.toList());
    }

    private List<DevKitConnectorTypeDefinitionDecorator> buildParameters(List<Parameter> parameters) {
        ImmutableList.Builder<DevKitConnectorTypeDefinitionDecorator> builder = ImmutableList.builder();

        for (Parameter parameter : parameters) {
            DevKitConnectorTypeDefinitionDecorator typeDefinitionDecorator = new DevKitConnectorTypeDefinitionDecorator(parameter.getExternalName(), parameter.getInternalName(), parameter.getTypeDefinition(), basePackage, pojoFQNamePool);
            if(typeDefinitionDecorator.requiresPojo() && StringUtils.isBlank(typeDefinitionDecorator.getClassName())){
                typeDefinitionDecorator.setClassName(capitalize(getMethodName()) + capitalize(parameter.getInternalName()));
            }
            builder.add(typeDefinitionDecorator);
        }

        return builder.build();
    }

    private List<DevKitConnectorTypeDefinitionDecorator> buildDecoratedParameters() {
        return ImmutableList.<DevKitConnectorTypeDefinitionDecorator> builder()
                .addAll(uriParameters)
                .addAll(queryParameters)
                .addAll(headers)
                .addAll(partParameters)
                .build();
    }

    public String getMethodName() {
        return getJavaName(operation.getCanonicalName());
    }

    public String getTestClassName() {
        return capitalize(getMethodName()) + "TestCase";
    }

    public String getArguments() {
        List<String> stream = new ArrayList<>();

        // Parses the query params.
        for (DevKitConnectorTypeDefinitionDecorator parameter : getDecoratedParameters()) {
            StringBuilder sb = new StringBuilder();

            if(StringUtils.isNotBlank(parameter.getAnnotatedParameterName())){
                sb.append("@FriendlyName(\"");
                sb.append(StringEscapeUtils.escapeJava(parameter.getAnnotatedParameterName()));
                sb.append("\") ");
            }
            else if(!isValidDevKitParameterName(parameter.getInternalName())){
                sb.append("@FriendlyName(\"");
                sb.append(StringEscapeUtils.escapeJava(parameter.getInternalName()));
                sb.append("\") ");
            }

            if(!parameter.isRequired()){
                sb.append("@Optional ");
            }

            sb.append(buildDefaultValueArgumentAnnotation(parameter));

            sb.append(parameter.getClassName());
            sb.append(" ");
            sb.append(parameter.getInternalName());
            sb.append(" ");
            stream.add(sb.toString());
        }

        // Parses the content entity.
        if (this.operation.getInputMetadata() != null) {
            StringBuilder sb = new StringBuilder();
            sb.append("@RefOnly @Default(\"#[payload]\") ");
            sb.append(getDecoratedInputMetadata().getClassName());
            sb.append(" entity");
            stream.add(sb.toString());
        }

        return Joiner.on(", ").join(stream);
    }

    protected String buildDefaultValueArgumentAnnotation(DevKitConnectorTypeDefinitionDecorator parameter){
        if(!parameter.hasDefaultValue())
            return "";

        StringBuilder builder = new StringBuilder();
        builder.append("@Default(\"");

        if(parameter.isEnumType() && !parameter.getClassName().toLowerCase().endsWith("string")){
            RuleFactory ruleFactory = new RuleFactory();
            String jsonFieldName = ruleFactory.getNameHelper().getFieldName(parameter.getDefaultValue(), null);
            jsonFieldName = ruleFactory.getNameHelper().replaceIllegalCharacters(jsonFieldName);
            jsonFieldName = ruleFactory.getNameHelper().normalizeName(jsonFieldName);
            jsonFieldName = Character.toLowerCase(jsonFieldName.charAt(0)) + jsonFieldName.substring(1);
            String enumJsonFieldName = splitNumbersAndLetters(splitCapsWithUnderscores(jsonFieldName), "_").toUpperCase();

            builder.append(enumJsonFieldName);
        }
        else{
            builder.append(StringEscapeUtils.escapeJava(parameter.getDefaultValue()));
        }

        builder.append("\") ");
        return builder.toString();
    }

    public DevKitConnectorTypeDefinitionDecorator getDecoratedInputMetadata() {
        return inputMetadata;
    }

    public DevKitConnectorTypeDefinitionDecorator getDecoratedOutputMetadata() {
        return outputMetadata;
    }

    public List<DevKitConnectorTypeDefinitionDecorator> getDecoratedQueryParameters() {
        return queryParameters;
    }

    public List<DevKitConnectorTypeDefinitionDecorator> getDecoratedUriParameters() {
        return uriParameters;
    }

    public List<DevKitConnectorTypeDefinitionDecorator> getDecoratedHeaders() {
        return headers;
    }

    public List<DevKitConnectorTypeDefinitionDecorator> getDecoratedParameters() {
        return parameters;
    }

    public List<DevKitSecuritySchemeDecorator> getSecuritySchemes() {
        return securitySchemes;
    }

}
