package org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser;

import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinitionBuilder;
import org.mule.connectivity.restconnect.internal.model.typesource.*;
import org.mule.connectivity.restconnect.internal.model.typesource.PrimitiveTypeSource.PrimitiveType;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.predicate.HasBodyPredicate;
import org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.predicate.IsSupportedTypePredicate;
import org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.predicate.OkResponsePredicate;
import org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils;
import org.raml.v2.api.model.v10.bodies.Response;
import org.raml.v2.api.model.v10.datamodel.*;
import org.raml.v2.api.model.v10.methods.Method;

import javax.ws.rs.core.MediaType;
import java.util.*;

import static org.apache.commons.lang3.StringUtils.trimToNull;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.PrimitiveMetadataModelFactory.createPrimitiveMetadataModel;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils.getPartParameterList;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils.isDefault;


public class RamlParserTypeDefinitionFactory {

    private static final Predicate<Response> okAndBodyPredicate = Predicates.and(new OkResponsePredicate(), new HasBodyPredicate());
    private static final Logger logger = LogManager.getLogger(RamlParserTypeDefinitionFactory.class);

    private final Method method;
    private final JsonSchemaPool jsonSchemaPool;

    public RamlParserTypeDefinitionFactory(Method method, JsonSchemaPool jsonSchemaPool) {
        this.jsonSchemaPool = jsonSchemaPool;
        this.method = method;
    }

    public TypeDefinition constructInputMetadata() throws Exception {
        return getTypeDefinitionWithMediaType(method.body());
    }

    public TypeDefinition constructOutputMetadata() throws Exception {
        List<TypeDeclaration> mediaTypes = new ArrayList<>();
        for (Response response : Iterables.filter(method.responses(), okAndBodyPredicate)) {
            mediaTypes.addAll(Collections2.filter(response.body(), new IsSupportedTypePredicate()));
        }

        return getTypeDefinitionWithMediaType(mediaTypes);
    }

    private TypeDefinition getTypeDefinitionWithMediaType(List<TypeDeclaration> typeDeclarations) throws Exception {
        Collection<TypeDeclaration> supportedTypeDeclarations = getTypeDeclarations(typeDeclarations);
        if (supportedTypeDeclarations == null)
            return null;

        Optional<TypeDeclaration> defaultTypeDeclaration = supportedTypeDeclarations.stream().filter(x -> isDefault(x)).findFirst();

        TypeDeclaration typeDeclaration = defaultTypeDeclaration.orElseGet(() -> supportedTypeDeclarations.iterator().next());

        return getTypeDefinition(typeDeclaration, typeDeclaration.name(), jsonSchemaPool);
    }

    public static TypeDefinition getTypeDefinition(TypeDeclaration typeDeclaration, JsonSchemaPool jsonSchemaPool) throws Exception {
        return getTypeDefinition(typeDeclaration, null, jsonSchemaPool);
    }

    public static TypeDefinition getTypeDefinition(TypeDeclaration typeDeclaration, String mediaType, JsonSchemaPool jsonSchemaPool) throws Exception {

        if(MediaType.MULTIPART_FORM_DATA.equalsIgnoreCase(mediaType)){
            if(typeDeclaration instanceof ObjectTypeDeclaration){
                ObjectTypeDeclaration objTypeDeclaration = ((ObjectTypeDeclaration)typeDeclaration);
                if(objTypeDeclaration.properties().size() > 0){
                    return buildTypeDefinition(mediaType, typeDeclaration, new MultipartTypeSource(getPartParameterList(objTypeDeclaration.properties(), new JsonSchemaPool()), new XmlTypeSource(objTypeDeclaration.toXmlSchema())));
                }
            }
            return buildTypeDefinition(mediaType, typeDeclaration, new PrimitiveTypeSource(PrimitiveType.STRING));
        }

        if(typeIsDefinedWithRAML(typeDeclaration)) {
            return safelyGetTypeDefinitionFromRAMLType(typeDeclaration, mediaType, jsonSchemaPool);
        }

        else if (typeIsDefinedWithXMLSchema(typeDeclaration)) {
            XMLTypeDeclaration xmlTypeDeclaration = (XMLTypeDeclaration) typeDeclaration;
            String elementName = StringUtils.isNotBlank(xmlTypeDeclaration.internalFragment()) ? xmlTypeDeclaration.internalFragment() : null;
            TypeSource source = new XmlTypeSource(xmlTypeDeclaration.schemaContent(), elementName, xmlTypeDeclaration.schemaPath());

            return buildTypeDefinition(mediaType, xmlTypeDeclaration, source);
        }

        else if (typeIsDefinedWithJSONSchema(typeDeclaration)) {
            JSONTypeDeclaration jsonTypeDeclaration = (JSONTypeDeclaration) typeDeclaration;

            String schemaContent = jsonTypeDeclaration.schemaContent();
            if(StringUtils.isNotBlank(schemaContent)){
                if(jsonSchemaPool.containsSchema(schemaContent)){
                    return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
                }
                else{
                    TypeSource source = new JsonTypeSource(jsonTypeDeclaration.schemaContent());
                    TypeDefinition newType = buildTypeDefinition(mediaType, jsonTypeDeclaration, source);
                    jsonSchemaPool.putSchemaTypeDefinitionPair(schemaContent, newType);
                    return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
                }
            }

            TypeSource source = new JsonTypeSource(jsonTypeDeclaration.schemaContent());

            return buildTypeDefinition(mediaType, jsonTypeDeclaration, source);
        }

        else if(typeIsEnum(typeDeclaration)){

            String schemaContent = typeDeclaration.toJsonSchema();
            if(StringUtils.isNotBlank(schemaContent)) {
                if(jsonSchemaPool.containsSchema(schemaContent)) {
                    return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
                }
                else{
                    TypeSource source = new JsonTypeSource(typeDeclaration.toJsonSchema());
                    TypeDefinition newType = buildTypeDefinition(mediaType, typeDeclaration, source);
                    jsonSchemaPool.putSchemaTypeDefinitionPair(schemaContent, newType);
                    return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
                }
            }

            TypeSource source = new JsonTypeSource(typeDeclaration.toJsonSchema());
            return buildTypeDefinition(mediaType, typeDeclaration, source);
        }

        else if (typeIsPrimitive(typeDeclaration)) {
            return buildTypeDefinition(mediaType, typeDeclaration, createPrimitiveMetadataModel(typeDeclaration));
        }

        else if (typeDefinitionIsNotProvided(typeDeclaration)) {
            return buildTypeDefinition(mediaType, typeDeclaration, new PrimitiveTypeSource(PrimitiveType.STRING));
        }

        // This shouldn't happen as it's already validated that the declaration is supported.
        else {
            logger.warn("Type {} cannot be parsed.", typeDeclaration.type());
        }

        return null;
    }

    private static TypeDefinition buildTypeDefinition(String mediaType, TypeDeclaration typeDeclaration, TypeSource source){
        List<String> enumValues = null;
        if(typeDeclaration instanceof StringTypeDeclaration){
            enumValues = ((StringTypeDeclaration)typeDeclaration).enumValues();
        }

        return new TypeDefinitionBuilder(mediaType == null ? null : MediaType.valueOf(mediaType), source, typeDeclaration.required(), typeDeclaration  instanceof ArrayTypeDeclaration, typeDeclaration instanceof UnionTypeDeclaration)
                .withDescription(typeDeclaration.description() == null ? null : trimToNull(typeDeclaration.description().value()))
                .withDefaultValue(getTypeDeclarationDefaultValue(typeDeclaration))
                .withExample(typeDeclaration.example() == null ? null : typeDeclaration.example().value())
                .withAnnotatedDisplayName(RamlParserUtils.getAnnotatedParameterName(typeDeclaration))
                .withEnumValues(enumValues)
                .build();
    }

    private static String getTypeDeclarationDefaultValue(TypeDeclaration typeDeclaration){
        String defaultValue = typeDeclaration.defaultValue();

        return StringUtils.isNotBlank(defaultValue) ? defaultValue : null;
    }

    private static TypeDefinition safelyGetTypeDefinitionFromRAMLType(TypeDeclaration typeDeclaration, String mediaType, JsonSchemaPool jsonSchemaPool) throws Exception {
        // The XML SDK won't support RAML types until MULE-11501 is done.
        // There was a workaround implemented for translating those into JSON schemas implemented in RESTC-156 and RESTC-164.
        // There are known cases that are failing due to inconsistencies in the transformation between RAML types and
        // JSON schemas, so this is for preventing that the Smart Connector can be generated even though a single type
        // throws an exception.
        try {

            String schemaContent = typeDeclaration.toJsonSchema();
            if(StringUtils.isNotBlank(schemaContent)) {
                if (jsonSchemaPool.containsSchema(schemaContent)) {
                    return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
                }
                else{
                    TypeDefinition newType = buildTypeDefinition(mediaType, typeDeclaration, new JsonTypeSource(typeDeclaration.toJsonSchema()));
                    jsonSchemaPool.putSchemaTypeDefinitionPair(schemaContent, newType);
                    return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
                }
            }

            return buildTypeDefinition(mediaType, typeDeclaration, new JsonTypeSource(typeDeclaration.toJsonSchema()));
        }

        catch (RuntimeException exception) {
            logger.warn("Couldn't generate a type definition for " + typeDeclaration.name(), exception);

            // In the future this should be changed to type "any" (RESTC-176).
            return buildTypeDefinition(mediaType, typeDeclaration, new PrimitiveTypeSource(PrimitiveType.STRING));
        }
    }

    private Collection<TypeDeclaration> getTypeDeclarations(List<TypeDeclaration> typeDeclarations) {
        if (typeDeclarations == null) {
            return null;
        }

        Collection<TypeDeclaration> supportedTypeDeclarations = Collections2.filter(typeDeclarations, new IsSupportedTypePredicate());

        if (supportedTypeDeclarations.isEmpty())
            return null;

        if (supportedTypeDeclarations.size() > 1) {
            logger.warn("Operation {} {} contains more than one media type. Found: {} media types. Will use the first one.",
                    method.resource().resourcePath(), method.method(), typeDeclarations.size());
        }

        return supportedTypeDeclarations;
    }

    public static boolean typeIsDefinedWithRAML(TypeDeclaration typeDeclaration) {
        return typeDeclaration instanceof ObjectTypeDeclaration || typeDeclaration instanceof ArrayTypeDeclaration || typeDeclaration instanceof UnionTypeDeclaration;
    }

    public static boolean typeIsDefinedWithXMLSchema(TypeDeclaration typeDeclaration) {
        return typeDeclaration instanceof XMLTypeDeclaration;
    }

    public static boolean typeIsDefinedWithJSONSchema(TypeDeclaration typeDeclaration) {
        return typeDeclaration instanceof JSONTypeDeclaration;
    }

    public static boolean typeIsPrimitive(TypeDeclaration typeDeclaration) {
        return typeDeclaration instanceof BooleanTypeDeclaration ||
                typeDeclaration instanceof DateTimeOnlyTypeDeclaration ||
                typeDeclaration instanceof DateTypeDeclaration ||
                typeDeclaration instanceof DateTimeTypeDeclaration ||
                typeDeclaration instanceof FileTypeDeclaration ||
                typeDeclaration instanceof IntegerTypeDeclaration ||
                typeDeclaration instanceof NumberTypeDeclaration ||
                typeDeclaration instanceof StringTypeDeclaration ||
                typeDeclaration instanceof TimeOnlyTypeDeclaration;
    }

    public static boolean typeIsEnum(TypeDeclaration typeDeclaration) {
        return typeDeclaration instanceof StringTypeDeclaration
               && ((StringTypeDeclaration)typeDeclaration).enumValues() != null
               && ((StringTypeDeclaration)typeDeclaration).enumValues().size() > 0;
    }

    // This is for the case that the type is not defined (for instance, when there is an example but not a definition).
    public static boolean typeDefinitionIsNotProvided(TypeDeclaration typeDeclaration) {
        return typeDeclaration instanceof AnyTypeDeclaration;
    }

}
