package org.mule.connectivity.restconnect.internal.templateEngine.decorator.type;

import org.mule.connectivity.restconnect.internal.model.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.model.typesource.JsonTypeSource;
import org.mule.connectivity.restconnect.internal.model.typesource.MultipartTypeSource;
import org.mule.connectivity.restconnect.internal.model.typesource.PrimitiveTypeSource;
import org.mule.connectivity.restconnect.internal.model.typesource.XmlTypeSource;
import org.mule.connectivity.restconnect.internal.util.FileGenerationUtils;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils.getXmlName;

import org.apache.commons.lang3.StringUtils;


public class SmartConnectorTypeDefinitionDecorator extends TypeDefinitionDecorator {

    private static final String DEFAULT_JSON_MEDIA_TYPE = "application/json";
    private static final String DEFAULT_XML_MEDIA_TYPE = "application/xml";
    private static final String DEFAULT_MULTIPART_MEDIA_TYPE = "application/xml";
    private static final String FORM_URLENCODED_MEDIA_TYPE = "application/x-www-form-urlencoded";

    private String outputSchemaFilename;
    private String outputSchemaFormat;
    private String name;
    private String type;
    private boolean isPassword;

    private static final List<String> RESERVED_SMART_CONNECTOR_PARAMETER_NAMES = new LinkedList<>(
        Arrays.asList("target")
    );

    public SmartConnectorTypeDefinitionDecorator(TypeDefinition typeDefinition) {
        super(typeDefinition);
        this.isPassword = false;
    }

    public SmartConnectorTypeDefinitionDecorator(String externalName, String internalName, TypeDefinition typeDefinition, boolean isPassword) {
        super(externalName, internalName, typeDefinition);
        this.isPassword = isPassword;
    }

    public String getExternalName() {
        if(this.name != null)
            return this.name;

        return super.getExternalName();
    }

    public String getInternalName() {
        String returnName;

        if(StringUtils.isNotBlank(this.name)){
            returnName = this.name;
        }
        else{
            returnName = getXmlName(super.getInternalName());
        }

        if(RESERVED_SMART_CONNECTOR_PARAMETER_NAMES.contains(returnName)){
            return "s" + returnName;
        }

        return returnName;
    }

    public String getPropertyName() {
        if(this.name != null)
            return this.name;

        return "property_" + getInternalName();
    }

    public String getType() {
        if (isPrimitiveType())
            return getMule4PrimitiveType();

        if(this.type != null)
            return this.type;

        return getInternalName();
    }

    public void setName(String name) {
        this.name = name;
    }

    public void setType(String type) {
        this.type = type;
    }

    private String getMule4PrimitiveType() {
        PrimitiveTypeSource typeSource = (PrimitiveTypeSource) getTypeDefinition().getSource();
        switch (typeSource.getType()) {
            case INTEGER:
            case NUMBER:
                return "number";
            case BOOLEAN:
                return "boolean";
            case DATE:
            case DATE_TIME:
            case DATE_ONLY:
            case DATE_TIME_ONLY:
                return "date";
            case TIME_ONLY:
                return "time";
            case STRING:
                return "string";
            case FILE:
                return "binary";
        }

        throw new IllegalArgumentException("Invalid type value: " + typeSource.toString());
    }

    public boolean requiresCatalog() {
        return !isPrimitiveType();
    }

    public String getOutputSchemaFilename() {
        return this.outputSchemaFilename;
    }

    public String getOutputSchemaFormat(){
        return this.outputSchemaFormat;
    }

    @Override
    public boolean hasMediaType(){
        return this.getMediaType() != null;
    }

    @Override
    public String getMediaType(){
        //Using source media type as default to avoid errors when unsupported media types are defined. (ie. text/plain)
        String mediaType = getMediaTypeForSource();

        if(mediaType == null && this.typeDefinition.getMediaType() != null){
            return this.typeDefinition.getMediaType().toString();
        }

        return mediaType;
    }

    private String getMediaTypeForSource(){
        if(this.getTypeDefinition().getSource() instanceof MultipartTypeSource)
            return DEFAULT_MULTIPART_MEDIA_TYPE;

        if(this.getTypeDefinition().getSource() instanceof XmlTypeSource)
            return DEFAULT_XML_MEDIA_TYPE;

        if(this.getTypeDefinition().getSource() instanceof JsonTypeSource)
            return DEFAULT_JSON_MEDIA_TYPE;

        return null;
    }

    public String getMediaTypeForTransformation(){
        //Using URL Encoded only in DW transformation since the catalog does not support the type.
        if ((this.typeDefinition.getMediaType() != null) &&
                this.getTypeDefinition().getMediaType().toString().equals(FORM_URLENCODED_MEDIA_TYPE))
            return FORM_URLENCODED_MEDIA_TYPE;

        return getMediaType();
    }

    public Path writeSchema(Path outputDir) throws IOException {
        this.outputSchemaFilename = FileGenerationUtils.writeSchema(this.getTypeDefinition().getSource(), getInternalName() + "-schema", outputDir);
        this.outputSchemaFormat = FileGenerationUtils.getSchemaFormat(this.getTypeDefinition().getSource());

        return outputDir.resolve(outputSchemaFilename);
    }

    public boolean hasElementName() {
        return this.getTypeDefinition().getSource() instanceof XmlTypeSource
                || this.getTypeDefinition().getSource() instanceof MultipartTypeSource;
    }

    public String getElementName() {
        if(this.getTypeDefinition().getSource() instanceof XmlTypeSource){
            return ((XmlTypeSource) this.getTypeDefinition().getSource()).getElementName();
        }
        else if(this.getTypeDefinition().getSource() instanceof MultipartTypeSource){
            return ((MultipartTypeSource) this.getTypeDefinition().getSource()).getXmlSourceElementName();
        }

        throw new IllegalArgumentException("This element does not have an element name");
    }

    public boolean isPassword() {
        return isPassword;
    }

    public void setPassword(boolean password) {
        isPassword = password;
    }


}
