package org.mule.connectivity.restconnect.internal.util;

import com.sun.codemodel.JCodeModel;
import com.sun.codemodel.JType;
import com.sun.tools.xjc.api.Mapping;
import com.sun.tools.xjc.api.S2JJAXBModel;
import com.sun.tools.xjc.api.SchemaCompiler;
import com.sun.tools.xjc.api.XJC;
import org.apache.commons.lang3.StringUtils;
import org.jsonschema2pojo.*;
import org.jsonschema2pojo.rules.RuleFactory;
import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.internal.model.typesource.JsonTypeSource;
import org.mule.connectivity.restconnect.internal.model.typesource.MultipartTypeSource;
import org.mule.connectivity.restconnect.internal.model.typesource.TypeSource;
import org.mule.connectivity.restconnect.internal.model.typesource.XmlTypeSource;
import org.xml.sax.InputSource;

import java.io.*;
import java.net.URL;
import java.nio.file.Path;
import java.util.jar.JarEntry;
import java.util.zip.ZipOutputStream;

import static java.lang.String.format;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils.removeJavaNameUnwantedCharacters;

public class FileGenerationUtils {

    private static GenerationConfig config = new DefaultGenerationConfig() {

        @Override
        public boolean isGenerateBuilders() { // set config option by overriding method
            return true;
        }

        @Override
        public String getTargetVersion() {
            return "1.7";
        }

        @Override
        public boolean isIncludeAdditionalProperties(){
            return false;
        }

        @Override public boolean isIncludeHashcodeAndEquals() {
            return false;
        }
    };

    public static String generatePojo(TypeSource source, String opName, File outputDir, String basePackage) throws GenerationException {
        if(source instanceof XmlTypeSource)
            return generatePojo((XmlTypeSource) source, opName, outputDir, basePackage);

        if(source instanceof JsonTypeSource)
            return generatePojo((JsonTypeSource) source, opName, outputDir, basePackage);

        if(source instanceof MultipartTypeSource)
            return generatePojo(((MultipartTypeSource) source).getXmlSource(), opName, outputDir, basePackage);

        throw new IllegalArgumentException("Typesource doesn't support POJO generation.");
    }

    private static String generatePojo(XmlTypeSource source, String opName, File outputDir, String basePackage) throws GenerationException {

        try{
            SchemaCompiler sc = XJC.createSchemaCompiler();
            String pojoPackage = basePackage + ".pojo." + removeJavaNameUnwantedCharacters(opName).toLowerCase();

            InputSource inputSource;
            if(StringUtils.isNotBlank(source.getSchemaPath())){
                inputSource = new InputSource(source.getSchemaPath());

            }
            else {
                inputSource = getInputSourceWithMockFile(source);
            }

            sc.getOptions().parseArgument(new String[] { "-XautoNameResolution"}, 0);
            sc.parseSchema(inputSource);

            S2JJAXBModel model = sc.bind();
            sc.setDefaultPackageName(pojoPackage);
            S2JJAXBModel newModel = sc.bind();
            if(newModel != null){
                model = newModel;
            }

            JCodeModel jCodeModel = model.generateCode(null, null);
            jCodeModel.build(outputDir);

            return findBestMatchClassName(model, source.getElementName());
        }
        catch (Throwable e){
            throw new GenerationException("Error generating Pojo from xsd", e);
        }
    }

    public static String findBestMatchClassName(S2JJAXBModel model, String elementName){
        for(Mapping mapping : model.getMappings()){
            if(mapping.getElement().getLocalPart().equalsIgnoreCase(elementName)){
                return mapping.getType().getTypeClass().fullName();
            }
        }

        return model.getMappings().iterator().next().getType().getTypeClass().fullName();
    }

    private static InputSource getInputSourceWithMockFile(XmlTypeSource source) throws IOException {
        File temp = File.createTempFile("schema", ".xsd");

        BufferedWriter bw = new BufferedWriter(new FileWriter(temp));
        bw.write(source.getValue());
        bw.close();

        final FileInputStream schemaStream = new FileInputStream(temp);
        final InputSource is = new InputSource(schemaStream);

        if(StringUtils.isNotBlank(source.getElementName())){
            is.setSystemId(temp.toURI().toString() + "#" + source.getElementName());
        }
        else{
            is.setSystemId(temp.toURI().toString());
        }

        return is;
    }

    private static String generatePojo(JsonTypeSource source, String opName, File outputDir, String basePackage) throws GenerationException {
        try{

            SchemaMapper mapper = new SchemaMapper(new RuleFactory(config, new Jackson2Annotator(config), new SchemaStore()), new SchemaGenerator());
            JCodeModel codeModel = new JCodeModel();
            JType generatedModel = mapper.generate(codeModel, opName, basePackage + ".pojo." + removeJavaNameUnwantedCharacters(opName).toLowerCase(), writeTempSchema(source.getValue(), opName));
            codeModel.build(outputDir);

            return generatedModel.fullName();
        }
        catch (Throwable e){
            throw new GenerationException("Error generating Pojo from JSON", e);
        }
    }

    private static URL writeTempSchema(String schemaContent, String operationName) throws IOException{
        //There is an issue in jsonschema2pojo where it will throw an stack overflow exception when not referencing a file
        //This code exists because of that
        //Check RESTC-459 / RESTC-455 for reference.
        File tempDir = new File(System.getProperty("java.io.tmpdir"));
        File tempFile = File.createTempFile(operationName, ".tmp", tempDir);
        FileWriter fileWriter = new FileWriter(tempFile, true);
        BufferedWriter bw = new BufferedWriter(fileWriter);
        bw.write(schemaContent);
        bw.close();
        return tempFile.toURI().toURL();
    }

    public static String writeSchema(TypeSource source, String name, Path outputDir) {
        try{
            String filename = name + "." + getExtension(source);

            try{
                writeStringToFile(outputDir.resolve(filename).toFile(), source.getValue());
            }
            catch (FileNotFoundException e){
                //This is probably happening because the filename is too long, so we try to fix this issue
                return writeSchema(source, generateSmallerFileNameOrFail(filename, e), outputDir);
            }

            return filename;
        }
        catch (IOException e){
                throw new IllegalArgumentException(format("Path %s is invalid.", outputDir), e);
        }
    }

    private static String generateSmallerFileNameOrFail(String filename, Exception cause) throws IOException{
        if(filename.length() > 15){
            return filename.substring(0, filename.length() - 10);
        }
        else{
            throw new IOException("Could not generate smaller filename", cause);
        }
    }

    private static String getExtension(TypeSource source) {
        if(source instanceof XmlTypeSource)
            return "xsd";

        if(source instanceof JsonTypeSource)
            return "json";

        if(source instanceof MultipartTypeSource)
            return "xsd";

        throw new IllegalArgumentException("Typesource doesn't support schema generation.");
    }

    public static String getSchemaFormat(TypeSource source) {
        if(source instanceof XmlTypeSource)
            return "application/xml+schema";

        if(source instanceof JsonTypeSource)
            return "application/json+schema";

        if(source instanceof MultipartTypeSource)
            return "application/xml+schema";

        throw new IllegalArgumentException("Typesource doesn't support schema generation.");
    }

    public static void generateJarFileFromDirectory(Path directory, Path output) throws Exception {
        generateJarArchiveFromDirectory(directory.toFile(), output.toFile());
    }

    private static void generateJarArchiveFromDirectory(File inputDirectory, File outputFile) throws Exception {
        FileOutputStream outputStream = new FileOutputStream(outputFile);
        BufferedOutputStream bufferedStream = new BufferedOutputStream(outputStream);

        ZipOutputStream zipOutputStream = new ZipOutputStream(bufferedStream);

        addToJar(zipOutputStream, inputDirectory, inputDirectory.getAbsolutePath().length());
        zipOutputStream.finish();
        zipOutputStream.close();
    }


    private static void addToJar(ZipOutputStream outputStream, File toBeJared, Integer directoryLength) throws Exception{
        if (toBeJared == null || !toBeJared.exists())
            return;

        String pathInZip = toBeJared.getAbsolutePath().substring(directoryLength, toBeJared.getAbsolutePath().length()).replace("\\", "/");

        if(pathInZip.startsWith("/")){
            pathInZip = pathInZip.substring(1, pathInZip.length());
        }

        if(toBeJared.isDirectory()){
            if (!pathInZip.isEmpty())
            {
                if (!pathInZip.endsWith("/"))
                    pathInZip += "/";
                JarEntry entry = new JarEntry(pathInZip);
                entry.setTime(toBeJared.lastModified());
                outputStream.putNextEntry(entry);
                outputStream.closeEntry();
            }
            for (File nestedFile: toBeJared.listFiles())
                addToJar(outputStream, nestedFile, directoryLength);
            return;
        }

        JarEntry entry = new JarEntry(pathInZip);
        entry.setTime(toBeJared.lastModified());
        outputStream.putNextEntry(entry);

        // Write file to archive
        byte[] buffer = new byte[12000];
        BufferedInputStream in = new BufferedInputStream(new FileInputStream(toBeJared));

        try{
            while (true)
            {
                int count = in.read(buffer);
                if (count == -1)
                    break;
                outputStream.write(buffer, 0, count);
            }
            outputStream.closeEntry();
        }
        finally{
            in.close();
        }
    }
}
